// Clock.cpp : Implementation of CClock
#include "stdafx.h"



#include "ClockFace.h"
#include "Clock.h"
#include "ocidl.h"	// Added by ClassView

#include "olectl.h"	// Added by ClassView

#include "about.h"

#define MIN_DATE                (-657434L)  // about year 100
#define MAX_DATE                2958465L    // about year 9999


// static BOOL CClockLicense::m_bLicenseValid = FALSE;



const int g_nWinderTimerId = 1;

const double g_rgdbSize [CClock::ecMaximum] = {

		0.0,
		0.5, 0.7, 0.6,
		0.05, 0.05, 0.0, 0.0,
		0.9, 0.9125, 0.95,
		0.0, 0.075, 0.0, 0.0,
		0.075, 0.0

};

const int g_rgnThickness [CClock::ecMaximum] = {

	0,
	4, 2, 1,
	0, 0, 1, 2,
	2, 1, 1,
	0, 0, 0, 0,
	1, 0

};


const CClockComponent::eDrawn g_rgDrawn [CClock::ecMaximum] = {

	CClockComponent::edHidden,
	CClockComponent::edFancy, CClockComponent::edFancy, CClockComponent::edFancy,
	CClockComponent::edDrawn, CClockComponent::edDrawn, CClockComponent::edFancy, 
		CClockComponent::edDrawn, 
	CClockComponent::edDrawn, CClockComponent::edDrawn, CClockComponent::edDrawn,
	CClockComponent::edDrawn, CClockComponent::edDrawn, CClockComponent::edDrawn,
		CClockComponent::edDrawn, 
	CClockComponent::edDrawn, CClockComponent::edHidden

};

const int g_nBack = -1;
const int g_nBorder = -2;
const int g_nFill = -3;
const int g_nFore = -4;

const int g_rgnDefault [CClock::ecMaximum] = {

	g_nBack,
	g_nFore, g_nFore, g_nFore, 
	g_nFore, g_nFore, g_nFore, g_nBorder,
	g_nBorder, g_nBorder, g_nBorder,
	g_nFill, g_nFore, g_nFore, g_nFore,
	g_nFore, g_nFill

};

const int g_rgnNormal [CClock::ecMaximum] = {

	COLOR_3DFACE,
	COLOR_BTNTEXT, COLOR_BTNTEXT, COLOR_BTNTEXT,
	COLOR_BTNTEXT, COLOR_BTNTEXT, COLOR_BTNTEXT, COLOR_BTNTEXT,
	COLOR_BTNTEXT, COLOR_BTNTEXT, COLOR_BTNTEXT,
	COLOR_GRAYTEXT, COLOR_BTNTEXT, COLOR_GRAYTEXT, COLOR_GRAYTEXT, 
	0, COLOR_3DFACE

};

const int g_rgnSelected [CClock::ecMaximum] = {

	COLOR_3DFACE,
	COLOR_HIGHLIGHTTEXT, COLOR_HIGHLIGHTTEXT, COLOR_HIGHLIGHTTEXT,
	COLOR_HIGHLIGHTTEXT, COLOR_HIGHLIGHTTEXT, COLOR_HIGHLIGHTTEXT, COLOR_BTNTEXT, 
	COLOR_BTNTEXT, COLOR_BTNTEXT, COLOR_BTNTEXT,
	COLOR_GRAYTEXT, COLOR_BTNTEXT, COLOR_GRAYTEXT, COLOR_GRAYTEXT,
	0, COLOR_3DFACE
};

const int g_rgnDisabled [CClock::ecMaximum] = {

	COLOR_3DFACE,
	COLOR_GRAYTEXT, COLOR_GRAYTEXT, COLOR_GRAYTEXT,
	COLOR_GRAYTEXT, COLOR_GRAYTEXT, COLOR_GRAYTEXT, COLOR_GRAYTEXT,
	COLOR_GRAYTEXT, COLOR_GRAYTEXT, COLOR_GRAYTEXT,
	COLOR_GRAYTEXT, COLOR_GRAYTEXT, COLOR_GRAYTEXT, COLOR_GRAYTEXT,
	0, COLOR_3DFACE
};



/////////////////////////////////////////////////////////////////////////////
// CClock



void CClock::Construct ()

{
	m_bDrawn = m_bEnable = false;
	long ltime;
	time (&ltime);

	struct tm *pTime = gmtime (&ltime);
	_ASSERTE (! IsBadReadPtr (pTime, sizeof (struct tm)));

	SetTimeSimply (pTime -> tm_hour, pTime -> tm_min, pTime -> tm_sec);

	m_szGravityWell.cx = m_szGravityWell.cy = 4;
	m_bCapture = m_bFocus = false;
	RECT rcEmpty;

	SetRectEmpty (&rcEmpty);

	for (int i = 0; i < ecMaximum; i++)
		m_Component [i].Set (	g_rgnNormal [i] ? GetSysColor (g_rgnNormal [i]) : RGB (0, 0, 0), 
								g_rgnSelected [i] ? GetSysColor (g_rgnSelected [i]) : RGB (0, 0, 0), 
								g_rgnDisabled [i] ? GetSysColor (g_rgnDisabled [i]) : RGB (0, 0, 0), 
								rcEmpty,
								g_rgnThickness [i], 
								g_rgDrawn [i], 
								g_rgdbSize [i]);

	m_rgnTimer [etNone] = 0;
	m_rgnTimer [etWindStart] = 1000;
	m_rgnTimer [etWindRepeat] = 50;

	m_ptCentre.x = m_ptCentre.y = 0;
	m_bControl = m_bShift = m_bChanged = false;

#if defined (NO_BSTR)
	_tcsncpy (m_szDigitalFormat, _T ("%X"), g_nMaxEditText);

#if defined (_REG)
	_tcsncpy (	m_szIPAddress, 
				_T ("https://www.dylanharris.org/prose/clock/index.shtml"),
				_MAX_PATH);
#else // _REG
	_tcsncpy (	m_szIPAddress, 
				_T ("https://www.dylanharris.org/prose/clock/index.shtml"),
				_MAX_PATH);
#endif // _REG

#else // NO_BSTR
	m_bstrDigitalFormat = CComBSTR (_T ("%X"));

#if defined (_REG)
	m_bstrIPAddress	= CComBSTR (_T ("https://www.dylanharris.org/prose/clock/index.shtml"));
#else // _REG
	m_bstrIPAddress	= CComBSTR (_T ("https://www.dylanharris.org/prose/clock/index.shtml"));
#endif // _REG
#endif // NO_BSTR

	m_clrOldBackColor =	   m_clrBackColor   = GetSysColor (COLOR_3DFACE);
	m_clrOldBorderColor =  m_clrBorderColor = GetSysColor (COLOR_BTNTEXT);
	m_clrOldFillColor =	   m_clrFillColor   = GetSysColor (COLOR_3DFACE);
	m_clrOldForeColor =	   m_clrForeColor   = GetSysColor (COLOR_BTNTEXT);

	m_nOldBorderWidth =		m_nBorderWidth = 1;
	m_nOldDrawMode = m_nDrawMode = 13;

	m_dwDrawMode = SRCCOPY;

	m_nBorderStyle = m_nBackStyle = m_nDrawStyle = m_nDrawWidth = 1;
	m_nFillStyle = m_nAppearance = 1;
	m_nMousePointer = 0;
	
#if defined (KEYBOARD)
	m_bEditText = false;

	if (! GetLocaleInfo (LOCALE_USER_DEFAULT, LOCALE_S1159, m_szAM, g_nID))
		m_szAM [0] = 0;

	if (! GetLocaleInfo (LOCALE_USER_DEFAULT, LOCALE_S2359, m_szPM, g_nID))
		m_szPM [0] = 0;
#endif // KEYBOARD
}




STDMETHODIMP CClock::get_Time(DATE * pVal)
{
	_ASSERTE (! IsBadWritePtr (pVal, sizeof (DATE)));

	if (! IsBadWritePtr (pVal, sizeof (DATE)))
		*pVal = m_Time;

	return S_OK;
}

STDMETHODIMP CClock::put_Time(DATE newVal)
{
	if (newVal < g_nMaxDate && newVal > g_nMinDate) {
	
		m_Time = newVal;
		m_UndoTime = m_Time;
		PerhapsFireViewChange ();

	}

	return S_OK;
}


void CClock::TimeFromDate (int& nHour, int& nMinute, int& nSecond)

{	double dblDate = static_cast <double> (m_Time);

	dblDate += ((dblDate > 0.0) ? g_dbHalfSecond : -g_dbHalfSecond);
	dblDate = fabs (dblDate);
	
	long nSecsInDay = static_cast <long> ((dblDate - floor (dblDate)) * 86400.);

	if (nSecsInDay == 0)
		nHour = nMinute = nSecond = 0;
	
	else {

		nSecond = static_cast <int> (nSecsInDay % 60L);
		long nMinutesInDay = nSecsInDay / 60L;
		nMinute = static_cast <int> (nMinutesInDay % 60);
		nHour = static_cast <int> (nMinutesInDay / 60);

	}

}


bool CClock::SetTimeSimply (const int nHour, const int nMinute, const int nSecond)

{
	if (nHour >= 0 && nHour < 24)
		if (nMinute >= 0 && nMinute < 60)
			if (nSecond >= 0 && nSecond < 60) {
				
				double dblTime = static_cast <double> (
					static_cast <long> (nHour) * 3600L +
					static_cast <long> (nMinute) * 60L +
					static_cast <long> (nSecond)) / 86400.;
				m_Time = static_cast <DATE> (dblTime);
				m_UndoTime = static_cast <DATE> (dblTime);
				return true;

			}

	return false;

}



STDMETHODIMP CClock::SetTime(int nHour, int nMinute, int nSecond)

{
	if (SetTimeSimply (nHour, nMinute, nSecond))
		PerhapsFireViewChange ();
					
	return S_OK;
}





STDMETHODIMP CClock::SetCurrentTime ()
{
	long ltime;
	time (&ltime);

	struct tm *pTime = gmtime (&ltime);
	_ASSERTE (! IsBadReadPtr (pTime, sizeof (struct tm)));

	SetTime (pTime -> tm_hour, pTime -> tm_min, pTime -> tm_sec);
	Fire_Change (pTime -> tm_hour, pTime -> tm_min, pTime -> tm_sec);
		
	return S_OK;
}




LRESULT CClock::OnMouseMove (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)

{	bool bIn = false;
	POINT pt;

	if (! IsWindowEnabled () || ! m_bDrawn)
		return 0;

	pt.x = LOWORD (lParam);
	pt.y = HIWORD (lParam);

	if (m_bDown)
		if (DragTime (pt))
			return 0;

	eComponent h = GetHotSpot (pt);

	if (InWindow (h)) {

		HWND hWndCapture = GetCapture ();
		bIn = (hWndCapture == NULL) || (hWndCapture == m_hWnd);

#if defined (_DEBUG)
		if (hWndCapture == m_hWnd)
			_ASSERT (m_bCapture);
#endif // _DEBUG

	}

	if (! bIn) {

		if (m_bCapture)	{

			ReleaseCapture ();
			m_bCapture = false;

		}

		bHandled = FALSE;
		return 0;

	}

	if (! m_bCapture) {

		SetCapture ();
		m_bCapture = true;

	}

	SetMouseCursor (h, pt);
    return 0;
}



STDMETHODIMP CClock::get_GravityWellX(long * pVal)
{
	_ASSERTE (! IsBadWritePtr (pVal, sizeof (long)));

	if (! IsBadWritePtr (pVal, sizeof (long)))
		*pVal = m_szGravityWell.cx;

	return S_OK;
}

STDMETHODIMP CClock::put_GravityWellX(long newVal)
{
	m_szGravityWell.cx = newVal;
	return S_OK;
}

STDMETHODIMP CClock::get_GravityWellY(long * pVal)
{
	_ASSERTE (! IsBadWritePtr (pVal, sizeof (long)));

	if (! IsBadWritePtr (pVal, sizeof (long)))
		*pVal = m_szGravityWell.cy;

	return S_OK;
}

STDMETHODIMP CClock::put_GravityWellY(long newVal)
{
	m_szGravityWell.cy = newVal;
	return S_OK;
}



CClock::eComponent CClock::GetHotSpot (const POINT pt)

{
	if (m_bDrawn && m_bEnable)
		for (int i = ecOutside + 1; i < ecMaximum; i++)
			if (m_Component [i].PtInRect (pt))
				return static_cast <eComponent> (i);

	return ecOutside;
}




HCURSOR CClock::SetMouseCursor (const eComponent h, const POINT pt)

{	int nHour, nMinute, nSecond;

	TimeFromDate (nHour, nMinute, nSecond);

	switch (h) {

		case ecHour :
			if (nHour > 11)
				nHour -= 12;

			return SetAppropriateCursor (nHour * 5);

		case ecMinute :
			return SetAppropriateCursor (nMinute);

		case ecSecond :
			return SetAppropriateCursor (nSecond);

		case ecCyberspace :
#if defined (NO_BSTR)
			if (! m_szIPAddress [0])
				break;
#else // NO_BSTR
			{	USES_CONVERSION;

				LPSTR lpszIPAddress = OLE2A (m_bstrIPAddress.m_str);

				if (lpszIPAddress == NULL || lpszIPAddress [0] == 0)
					break;

			}
#endif // NO_BSTR
			// drop through

		case ecAMPM :
		case ecWindUp :
		case ecWindDown :
			return SetCursor (LoadCursor (NULL, IDC_UPARROW));

#if defined (KEYBOARD)
		case ecDigital :
			return SetCursor (LoadCursor (NULL, IDC_IBEAM));
#endif // KEYBOARD

	}

	return SetCursor (LoadCursor (NULL, IDC_ARROW));

}




HCURSOR CClock::SetAppropriateCursor (const int nMinute)

{
	if (	(nMinute > 56) || 
			(nMinute < 6) ||
			(nMinute > 26 && nMinute < 34))
		return SetCursor (LoadCursor (NULL, IDC_SIZEWE));

	if (	(nMinute > 11 && nMinute < 19) ||
			(nMinute > 41 && nMinute < 49))
		return SetCursor (LoadCursor (NULL, IDC_SIZENS));

	if (	(nMinute < 15) ||
			(nMinute > 30 && nMinute < 45))
		return SetCursor (LoadCursor (NULL, IDC_SIZENWSE));

	return SetCursor (LoadCursor (NULL, IDC_SIZENESW));

}



LRESULT CClock::OnLButtonDown (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)

{	
	if (! IsWindowEnabled ())
		return 0;

	SetControlFocus (TRUE);

	POINT pt;
	pt.x = LOWORD (lParam);
	pt.y = HIWORD (lParam);

	m_bChanged = false;
	m_ecDown = GetHotSpot (pt);

	m_bControl = !! (wParam & MK_CONTROL);
	m_bShift = !! (wParam & MK_SHIFT);

	switch (m_ecDown) {

		case ecOutside :
			Uncapture ();
			bHandled = FALSE;
			Fire_LButtonDown (pt.x, pt.y);
			return TRUE;

		case ecAMPM :
			SwapAMPM ();
			break;

		case ecWindUp :
		case ecWindDown :
			ApplyWinder ();
			m_nWinderTimer = SetTimer (g_nWinderTimerId, m_rgnTimer [etWindStart], NULL);
			SetCursor (LoadCursor (NULL, IDC_UPARROW));
			break;

		case ecSecond :
		case ecMinute :
		case ecHour :
			SetCursor (LoadCursor (NULL, IDC_CROSS));
			break;

		case ecCyberspace :
			LegendLink ();
			break;

#if defined (KEYBOARD)
		case ecDigital :
			EditTimeText (etDown);
			break;
#endif // KEYBOARD

	}

	Fire_LButtonDown (pt.x, pt.y);
	m_bDown = true;
	return FALSE;
}



LRESULT CClock::OnLButtonUp (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)

{	
	if (! IsWindowEnabled ())
		return 0;

	POINT pt;
	pt.x = LOWORD (lParam);
	pt.y = HIWORD (lParam);

	m_bDown = false;
	eComponent h = GetHotSpot (pt);
	ZapTimers ();
	
	if (h == ecOutside) {

		Uncapture ();
		bHandled = FALSE;

	} else {

		m_ecDown = ecOutside;
		SetMouseCursor (h, pt);

	}

	Fire_LButtonUp (pt.x, pt.y);

	if (m_bChanged) {
		int nHour, nMinute, nSecond;

		TimeFromDate (nHour, nMinute, nSecond);
		Fire_Change (nHour, nMinute, nSecond);

	}

	return (h == ecOutside);
}




void CClock::SwapAMPM ()

{	int nHour, nMinute, nSecond;

	TimeFromDate (nHour, nMinute, nSecond);

	if (nHour < 12)
		nHour += 12;

	else
		nHour -= 12;

	SetTime (nHour, nMinute, nSecond);
	PerhapsFireViewChange ();
	Fire_Change (nHour, nMinute, nSecond);

}


LRESULT CClock::OnTimer (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)

{
	switch (wParam) {

		case g_nWinderTimerId :
			{	

				ApplyWinder ();

				KillTimer (g_nWinderTimerId);
				m_nWinderTimer = 
					SetTimer (wParam, m_rgnTimer [etWindRepeat], NULL);

				return FALSE;

			}

		default :
			bHandled = FALSE;
			break;

	}

	return TRUE;

}




void CClock::ApplyWinder ()

{
	if (	m_ecDown != ecWindUp &&
			m_ecDown != ecWindDown)
		return;

	if (m_bControl)
		IncrementTime (euHour, m_ecDown == ecWindUp);

	else
	if (m_bShift)
		IncrementTime (euSecond, m_ecDown == ecWindUp);

	else
		IncrementTime (euMinute, m_ecDown == ecWindUp);

}



void CClock::IncrementTime (const eUnit Unit, const bool bBack, const int nCount)

{	int nHour, nMinute, nSecond;

	TimeFromDate (nHour, nMinute, nSecond);

	if (! bBack) {

		if (Unit == euHour && m_Component [ecHour].IsShown ())
			nHour += nCount;

		else {
				
			if (Unit == euMinute && m_Component [ecMinute].IsShown ())
				nMinute += nCount;

			else
			if (m_Component [ecSecond].IsShown ()) {

				nSecond += nCount;

				while (nSecond > 59) {

					nMinute++;
					nSecond -= 60;

				}

			}

			while (nMinute > 59) {

				nHour++;
				nMinute -= 60;

			}

		}

		while (nHour > 23)	{

			Fire_NewDay (TRUE);
			nHour -= 24;

		}

	} else {

		if (Unit == euHour && m_Component [ecHour].IsShown ())
			nHour -= nCount;

		else {
				
			if (Unit == euMinute && m_Component [ecMinute].IsShown ())
				nMinute -= nCount;

			else
			if (m_Component [ecSecond].IsShown ()) {

				nSecond -= nCount;

				while (nSecond < 0) {

					nMinute--;
					nSecond += 60;

				}

			}

			while (nMinute < 0) {

				nHour--;
				nMinute += 60;

			}

		}

		while (nHour < 0) {

			Fire_NewDay (FALSE);
			nHour += 24;

		}

	}

	SetTime (nHour, nMinute, nSecond);
	PerhapsFireViewChange ();
	Fire_Update (nHour, nMinute, nSecond);
	m_bChanged = true;

}





void CClock::ZapTimers ()

{

	if (m_nWinderTimer) {

		KillTimer (g_nWinderTimerId);
		m_nWinderTimer = 0;

	}

}


bool CClock::DragTime (const POINT pt)

{	int nHour, nMinute, nSecond;

	TimeFromDate (nHour, nMinute, nSecond);

	switch (m_ecDown) {

		case ecHour :
			{	bool bPM = nHour > 11;

				if (bPM)
					nHour -= 12;

				nHour = DragHand (pt, nHour * 5) / 5;

				if (bPM)
					nHour += 12;

				else
				if (nHour >= 12)
					nHour -= 12;

			}

			break;
			
		case ecMinute :			
			nMinute = DragHand (pt, nMinute);
			break;
			
		case ecSecond :			
			nSecond = DragHand (pt, nSecond);
			break;

		default :
			return false;

	}

	SetTime (nHour, nMinute, nSecond);
	PerhapsFireViewChange ();
	Fire_Update (nHour, nMinute, nSecond);
	m_bChanged = true;
	return true;

}




int CClock::DragHand (const POINT pt, const int nPos)

{
	if (m_Component [ecCentre].PtInRect (pt))
		return nPos;

	int nXDiff = pt.x - m_ptCentre.x;
	int nYDiff = m_ptCentre.y - pt.y;

	bool bXNeg = nXDiff < 0;
	bool bYNeg = nYDiff < 0;

	nXDiff = abs (nXDiff);
	nYDiff = abs (nYDiff);

	double dbTan = atan2 (static_cast <double> (nXDiff), static_cast <double> (nYDiff));
	int nResult = static_cast <int> (dbTan / g_dbPi * 30.0);

	if (bXNeg && bYNeg)
		return nResult + 30;

	else
	if (bXNeg)
		return 60 - nResult;

	else
	if (bYNeg)
		return 30 - nResult;

	return nResult;

}






HRESULT CClock::OnDraw (ATL_DRAWINFO& di)

{
	CheckColours ();

	// zero current area information

	for (int i = 0; i < static_cast <int> (ecMaximum); i++)
		m_Component [i].SetRect ();

	// record control bounds

    RECT& rc = *(RECT*)di.prcBounds;

	SIZE szBounds;
	szBounds.cx = rc.right - rc.left;
	szBounds.cy = rc.bottom - rc.top;

	// Create off-screen DC and bitmap

	HBITMAP hBit, hOldBitmap;
    HPEN hPenBack, hOldPen;
	HBRUSH hBrushBack, hOldBrush;

	HDC hdc = CreateGDIObjects (di, hBit, hOldBitmap, hPenBack, hOldPen, hBrushBack, hOldBrush,
		szBounds);

	if (hdc == NULL)
		return E_OUTOFMEMORY;

	// draw background

	int nOldMode = SetBkMode (hdc, TRANSPARENT);

	if (m_nBackStyle)
		DrawBackground (hdc, rc, hBrushBack);

	// calculate position of face and size of winder

	RECT rcFace;
	SIZE szWinder;
	CalculateFacePosition (rc, rcFace, szWinder);

	// if outer circle is to be drawn, draw it.

	if (m_Component [ecCircle].IsShown ())
		DrawOuterCircle (hdc, rcFace, hPenBack);

	// if got focus...

	if (m_bFocus && m_Component [ecFocus].IsShown ())
		DrawFocus (hdc, rcFace, hPenBack);

	// calculate centre position and area

	int nWidth = rcFace.right - rcFace.left;
	int nHeight = rcFace.bottom - rcFace.top;

	m_ptCentre.x = (rcFace.left + rcFace.right) / 2;
	m_ptCentre.y = (rcFace.top + rcFace.bottom) / 2;

	m_Component [ecCentre].SetGravityRect (m_ptCentre, m_szGravityWell);

	// Give contact information

	DrawWebAddress (hdc, rcFace);

	// Get control time
	int nHour, nMinute, nSecond;
	TimeFromDate (nHour, nMinute, nSecond);

	// manage centre circle.

	DrawCentre (hdc, rcFace, hPenBack, hBrushBack);

	// if winder is to be drawn, draw it

	if (	m_Component [ecWinderArm].IsShown () ||
			m_Component [ecWindUp].IsShown () ||
			m_Component [ecWindDown].IsShown ())
		DrawWinder (hdc, rcFace, rc, szWinder, hPenBack, hBrushBack);

	if (	m_Component [ecQuarterGrad].IsShown () ||
			m_Component [ecFiveGrad].IsShown () ||
			m_Component [ecSingleGrad].IsShown ())
		DrawGraduations (hdc, rcFace, hPenBack);

	// AM/PM

	if (m_Component [ecAMPM].IsShown ())
		DrawAMPM (hdc, rcFace, nHour, nMinute, nSecond, hPenBack, hBrushBack);

	// Display time as text (if not editing)

#if ! defined (KEYBOARD)
	if (m_Component [ecDigital].IsShown ())
#else // KEYBOARD
	if (m_Component [ecDigital].IsShown () && ! m_bEditText)
#endif // KEYBOARD
		DrawTimeText (hdc, rcFace, nHour, nMinute, nSecond);

	// And finally...

	DrawHands (hdc, rcFace, hPenBack, nHour, nMinute, nSecond);

	// Display time as text (if editing)

#if defined (KEYBOARD)
	if (m_Component [ecDigital].IsShown () && m_bEditText)
		DrawTimeText (hdc, rcFace, nHour, nMinute, nSecond);
#endif // KEYBOARD

	// copy the clock bitmap onto the screen

	if (m_dwDrawMode != -1)
		BitBlt (di.hdcDraw, rc.left, rc.top, szBounds.cx, szBounds.cy, hdc, 0, 0, m_dwDrawMode);

	// clear up
	if (nOldMode)
		SetBkMode (hdc, nOldMode);

	DeleteGDIObjects (hdc, hBit, hOldBitmap, hPenBack, hOldPen, hBrushBack, hOldBrush);

	m_bDrawn = true;
	// et au revoir...
    return S_OK;
}




void CClock::DrawBackground (HDC hdc, const RECT& rcClient, HBRUSH hBrushBack)

{
	HBRUSH hBrushOut = m_Component [ecOutside].CreateSolidBrush ();

	if (hBrushOut != NULL) {

		SelectObject (hdc, hBrushOut);
		Rectangle (hdc, 0, 0, rcClient.right - rcClient.left, rcClient.bottom - rcClient.top);
		SelectObject (hdc, hBrushBack);
		DeleteObject (hBrushOut);

	}

}





HDC CClock::CreateGDIObjects (ATL_DRAWINFO& di, HBITMAP& hBit, HBITMAP& hOldBitmap,
		HPEN& hPenBack, HPEN& hOldPen, HBRUSH& hBrushBack, HBRUSH& hOldBrush,
		const SIZE szBounds)

{
	HDC hdc = CreateCompatibleDC (di.hdcDraw);

	if (hdc == NULL)
		return NULL;

	hOldBrush = NULL;

	hBit = CreateCompatibleBitmap (di.hdcDraw, szBounds.cx, szBounds.cy);

	if (hBit != NULL) {

		hOldBitmap = (HBITMAP) SelectObject (hdc, hBit);

		if (hOldBitmap != NULL) {

			hPenBack = m_Component [ecBackground].CreatePen (PS_SOLID);

			if (hPenBack != NULL) {

				hOldPen = (HPEN) SelectObject (hdc, hPenBack);

				if (hOldPen != NULL) {
					
					hBrushBack = m_Component [ecBackground].CreateSolidBrush ();

					if (hBrushBack != NULL)	{

						hOldBrush = (HBRUSH) SelectObject (hdc, hBrushBack);

						if (hOldBrush != NULL)
							return hdc;

					}

				}

			}

		}

	}

	DeleteGDIObjects (hdc, hBit, hOldBitmap, hPenBack, hOldPen, hBrushBack, hOldBrush);
	return NULL;
}



void CClock::DeleteGDIObjects (	HDC hdc, 
								const HBITMAP hBit, const HBITMAP hOldBitmap,
								const HPEN hPenBack, const HPEN hOldPen,
								const HBRUSH hBrushBack, const HBRUSH hOldBrush)

{
	// if bitmap and object creation failed, tidy up and exit

	if (hOldBrush != NULL)
		SelectObject (hdc, hOldBrush);

	if (hBrushBack != NULL)
		DeleteObject (hBrushBack);

	if (hOldPen != NULL)
		SelectObject (hdc, hOldPen);

	if (hPenBack != NULL)
		DeleteObject (hPenBack);

	if (hOldBitmap != NULL)
		SelectObject (hdc, hOldBitmap);

	if (hBit != NULL)
		DeleteObject (hBit);

	DeleteDC (hdc);

}



void CClock::CalculateFacePosition (const RECT& rcBounds, RECT& rcFace, SIZE& szWinder)

{
	rcFace.left = rcFace.top = 0;

	if (	! m_Component [ecWinderArm].IsShown () &&
			! m_Component [ecWindUp].IsShown () &&
			! m_Component [ecWindDown].IsShown ()) {

		rcFace.right = rcBounds.right - rcBounds.left;
		rcFace.bottom = rcBounds.bottom - rcBounds.top;
		szWinder.cx = szWinder.cy = 0;

	} else {

		szWinder.cx = (rcBounds.right - rcBounds.left) / 20;
		szWinder.cy = (rcBounds.bottom - rcBounds.top) / 20;

		rcFace.right = rcBounds.right - rcBounds.left - szWinder.cx;
		rcFace.top += szWinder.cy / 2;
		rcFace.bottom = rcBounds.bottom - rcBounds.top - szWinder.cy / 2;

	}

}


void CClock::DrawCentre (HDC hdc, const RECT& rcFace, const HPEN hPenBack, const HBRUSH hBrushBack)

{
	// calculate centre details

	m_ptCentre.x = (rcFace.left + rcFace.right) / 2;
	m_ptCentre.y = (rcFace.top + rcFace.bottom) / 2;

	int nWidth = static_cast <int> (static_cast <double> 
		(rcFace.right - rcFace.left) * m_Component [ecCentre].m_dbSize);
	int nHeight = static_cast <int> (static_cast <double> 
		(rcFace.bottom - rcFace.top) * m_Component [ecCentre].m_dbSize);

	m_Component [ecCentre].SetRect (
		m_ptCentre.x - nWidth / 2,
		m_ptCentre.y - nHeight / 2,
		m_ptCentre.x + nWidth / 2,
		m_ptCentre.y + nHeight / 2);

	// if centre circle is drawn, draw it

	if (m_Component [ecCentre].IsShown ()) {

		HPEN hPenCentre = m_Component [ecCentre].CreatePen (PS_SOLID, GetState (ecCentre));

		if (hPenCentre != NULL) {

			if (SelectObject (hdc, hPenCentre) != NULL) {

				HBRUSH hBrushCentre = m_Component [ecCentre].CreateSolidBrush (GetState (ecCentre));

				if (hBrushCentre != NULL) {

					if (SelectObject (hdc, hBrushCentre) != NULL) {

						Ellipse (hdc, 
							m_Component [ecCentre].m_rcIci.left,
							m_Component [ecCentre].m_rcIci.top,
							m_Component [ecCentre].m_rcIci.right,
							m_Component [ecCentre].m_rcIci.bottom);
						SelectObject (hdc, hBrushBack);

					}

					DeleteObject (hBrushCentre);

				}

				SelectObject (hdc, hPenBack);

			}

			DeleteObject (hPenCentre);

		}

	}

}



void CClock::DrawWinder (HDC hdc, const RECT& rcFace, const RECT rcBounds,
	const SIZE szWinder, const HPEN hPenBack, const HBRUSH hBrushBack)

{	int nJoin = (rcFace.right + rcBounds.right - rcBounds.left) / 2;

	m_Component [ecWinderArm].SetRect (
				rcFace.right, 
				m_ptCentre.y - szWinder.cy / 4,
				nJoin,
				m_ptCentre.y + szWinder.cy / 4);

	m_Component [ecWindUp].SetRect (
				nJoin, 
				m_ptCentre.y - szWinder.cy / 2,
				rcBounds.right - rcBounds.left,
				m_ptCentre.y);

	m_Component [ecWindDown].SetRect (
				nJoin, 
				m_ptCentre.y,
				rcBounds.right - rcBounds.left,
				m_ptCentre.y + szWinder.cy / 2);
	
	DrawRect (hdc, ecWinderArm, hPenBack, hBrushBack);
	DrawRect (hdc, ecWindUp, hPenBack, hBrushBack);
	DrawRect (hdc, ecWindDown, hPenBack, hBrushBack);

}




void CClock::DrawRect (HDC hdc, const eComponent c, const HPEN hPenBack, const HBRUSH hBrushBack)

{
	HPEN hPen = m_Component [c].CreatePen (PS_SOLID, GetState (c));

	if (hPen != NULL) {

		if (SelectObject (hdc, hPen) != NULL) {

			HBRUSH hBrush = m_Component [c].CreateSolidBrush (GetState (c));

			if (hBrush != NULL) {

				if (SelectObject (hdc, hBrush) != NULL) {

					Rectangle (hdc,
						m_Component [c].m_rcIci.left,
						m_Component [c].m_rcIci.top,
						m_Component [c].m_rcIci.right,
						m_Component [c].m_rcIci.bottom);
					SelectObject (hdc, hBrushBack);

				}

				DeleteObject (hBrush);

			}

			SelectObject (hdc, hPenBack);

		}

		DeleteObject (hPen);

	}

}



void CClock::DrawOuterCircle (HDC hdc, const RECT& rcFace, HPEN hPenBack)

{
	if (! m_nBorderStyle)
		return;

	HPEN hPenCircle = m_Component [ecCircle].CreatePen (PS_SOLID, GetState (ecCircle));

	if (hPenCircle == NULL)
		return;

	if (SelectObject (hdc, hPenCircle) != NULL) {

		Ellipse (hdc, rcFace.left, rcFace.top, rcFace.right, rcFace.bottom);
		SelectObject (hdc, hPenBack);

	}

	DeleteObject (hPenCircle);

}




void CClock::DrawGraduations (HDC hdc, const RECT& rcFace, const HPEN hPenBack)

{	
	HPEN hPenQuarterGrad = m_Component [ecQuarterGrad].CreatePen (PS_SOLID, GetState (ecQuarterGrad));

	if (hPenQuarterGrad != NULL) {

		HPEN hPenFiveGrad = m_Component [ecFiveGrad].CreatePen (PS_SOLID, GetState (ecFiveGrad));

		if (hPenFiveGrad != NULL) {

			HPEN hPenSingleGrad = m_Component [ecSingleGrad].CreatePen (PS_SOLID, GetState (ecSingleGrad));

			if (hPenSingleGrad != NULL) {

				for (int i = 0; i < 60; i++) {

					double dbFraction = m_Component [ecSingleGrad].m_dbSize;

					if (! (i % 15)) {
					
						if (! m_Component [ecQuarterGrad].IsShown ())
							continue;

						if (SelectObject (hdc, hPenQuarterGrad) == NULL)
							continue; 

						dbFraction = m_Component [ecQuarterGrad].m_dbSize;

					} else
					if (! (i % 5)) {
					
						if (! m_Component [ecFiveGrad].IsShown ())
							continue;

						if (SelectObject (hdc, hPenFiveGrad) == NULL)
							continue; 

						dbFraction = m_Component [ecFiveGrad].m_dbSize;

					} else {

						if (! m_Component [ecSingleGrad].IsShown ())
							continue;

						if (SelectObject (hdc, hPenSingleGrad) == NULL)
							continue; 

						dbFraction = m_Component [ecSingleGrad].m_dbSize;

					}

					POINT ptFrom = PointInCircle (i, rcFace);
					POINT ptTo = PointInCircle (i, rcFace, dbFraction);

					MoveToEx (hdc, ptFrom.x, ptFrom.y, NULL);
					LineTo (hdc, ptTo.x, ptTo.y);

				}

				SelectObject (hdc, hPenBack);
				DeleteObject (hPenSingleGrad);

			}

			DeleteObject (hPenFiveGrad);
		}

		DeleteObject (hPenQuarterGrad);

	}

}








HFONT CClock::CreateHFont (HDC hDC, const int nHeight)

{
	if (hDC == NULL)
		return NULL;

	if (m_pFont == NULL)
		return CreateFont (nHeight, 0, 0, 0, FW_LIGHT, FALSE,
			FALSE, FALSE, DEFAULT_CHARSET, OUT_TT_PRECIS, CLIP_DEFAULT_PRECIS,
			DEFAULT_QUALITY, DEFAULT_PITCH | FF_DONTCARE, _T ("Arial"));

	IFont * pFont = NULL;
	m_pFont -> QueryInterface (IID_IFont, (void **)&pFont);

	if (pFont == NULL)
		return NULL;

	LOGFONT lf;
	ZeroMemory(&lf, sizeof(lf));

	CComBSTR bstr;
	pFont -> get_Name(&bstr);
	int j = 0;

	for (unsigned i = 0; i < bstr.Length (); i++)
		if (bstr.m_str [i] != 0)
			lf.lfFaceName [j++] = (char) bstr.m_str [i];

	lf.lfFaceName [j] = 0; 
//	lstrcpy(lf.lfFaceName, (const char *) bstr.m_str);

	CY cy;
	pFont -> get_Size(&cy);

	lf.lfHeight = nHeight;

	short sVal;
	pFont -> get_Weight(&sVal);
	lf.lfWeight  = sVal; 

	BOOL bFlag;
	pFont -> get_Italic(&bFlag);
	lf.lfItalic = bFlag; 

	pFont -> get_Underline(&bFlag);
	lf.lfUnderline = bFlag; 

	pFont -> get_Strikethrough(&bFlag);
	lf.lfStrikeOut = bFlag; 

	pFont -> get_Charset(&sVal);
	lf.lfCharSet = (unsigned char) sVal;
	 
	HFONT hFont = CreateFontIndirect(&lf);

	pFont -> Release();
	return hFont;
}


void CClock::DrawFaceText (HDC hdc, const eComponent e, const int nHeight,
#if ! defined (KEYBOARD)
	LPCTSTR lpszText, const int nLength, LPSIZE lpSize)

{
	COLORREF crOld = SetTextColor (hdc, m_Component [e].GetColour ());

	if (crOld == CLR_INVALID)
		return;
#else // KEYBOARD
	LPCTSTR lpszText, const int nLength, LPSIZE lpSize, const bool bSpecial)

{	COLORREF crOld;

	if (! bSpecial) {

		crOld = SetTextColor (hdc, m_Component [e].GetColour ());

		if (crOld == CLR_INVALID)
			return;

	}
#endif // KEYBOARD
	UINT uDrawFlags = DT_CENTER | DT_TOP | DT_NOCLIP | DT_NOPREFIX | DT_SINGLELINE;
	HFONT hFont = CreateHFont (hdc, nHeight);					
	HFONT hOldFont = NULL;

	if (hFont == NULL)
		uDrawFlags |= DT_INTERNAL;

	else
		hOldFont = (HFONT) SelectObject (hdc, hFont);

	if (lpSize != NULL)
		GetTextExtentPoint32 (hdc, lpszText, nLength, lpSize);

#if defined (KEYBOARD)
	else
#endif // KEYBOARD
	DrawText (hdc, lpszText, nLength, &m_Component [e].m_rcIci, uDrawFlags);

#if defined (KEYBOARD)
	if (! bSpecial)
#endif // KEYBOARD
		SetTextColor (hdc, crOld);

	if (hOldFont != NULL)				 
		SelectObject (hdc, hOldFont);

	if (hFont != NULL)
		DeleteObject (hFont);

}


int CClock::TimeToText (TCHAR szTime [], const int nMax,
	const int nHour, const int nMinute, const int nSecond, const bool bDefault)

{	long ltime;
	time (&ltime);
	struct tm *pTime = gmtime (&ltime);

	pTime -> tm_sec = nSecond;
	pTime -> tm_min = nMinute;
	pTime -> tm_hour = nHour;

#if defined (NO_BSTR)
	return strftime (szTime, nMax, m_szDigitalFormat, pTime);
#elif defined (NOO2T)
	TCHAR szFormat [g_nMaxEditText];

	for (unsigned i = 0; i < __min (g_nMaxEditText - 1, m_bstrDigitalFormat.Length ()); i++)
		szFormat [i] = (TCHAR) m_bstrDigitalFormat.m_str [i];

	szFormat [i] = 0;

	return _tcsftime (szTime, nMax, szFormat, pTime);
#else // NOO2T
	if (bDefault)
		return strftime (szTime, nMax, _T ("%H:%M:%S"), pTime);

	USES_CONVERSION;

	return strftime (szTime, nMax, OLE2T (m_bstrDigitalFormat.m_str), pTime);
#endif // NOO2T
}




void CClock::DrawTimeText (HDC hdc, const RECT& rcFace, 
	const int nHour, const int nMinute, const int nSecond)

{
	TCHAR szTime [g_nMaxEditText];

	int nLen = TimeToText (szTime, g_nMaxEditText, nHour, nMinute, nSecond);

	int nHeight = rcFace.bottom - rcFace.top;

	SIZE Size;
	Size.cx = Size.cy = 0;

#if defined (KEYBOARD)
	DrawFaceText (hdc, ecDigital, nHeight / 8, szTime, nLen, &Size);
#endif // KEYBOARD

	m_Component [ecDigital].SetRect (
		m_ptCentre.x - Size.cx / 2, m_ptCentre.y + nHeight / 12, 
		m_ptCentre.x + Size.cx / 2, rcFace.bottom);

#if ! defined (KEYBOARD)
	DrawFaceText (hdc, ecDigital, nHeight / 8, szTime, nLen);
#else // KEYBOARD
	if (m_bEditText) {

		RECT rc;

		int nHeight =  (m_Component [ecDigital].m_rcIci.bottom -
						m_Component [ecDigital].m_rcIci.top) / 4;

		rc.left = m_Component [ecDigital].m_rcIci.left - nHeight;
		rc.top = m_Component [ecDigital].m_rcIci.top - nHeight;
		rc.right = m_Component [ecDigital].m_rcIci.right + nHeight;
		rc.bottom = m_Component [ecDigital].m_rcIci.bottom + nHeight;

		HBRUSH hBrush = CreateSolidBrush (GetSysColor (COLOR_WINDOW));
		HBRUSH hOldBrush = static_cast <HBRUSH> (SelectObject (hdc, hBrush));

		HPEN hPen = CreatePen (PS_SOLID, 1, GetSysColor (COLOR_WINDOWFRAME));
		HPEN hOldPen = static_cast <HPEN> (SelectObject (hdc, hOldPen));

		Rectangle (hdc, rc.left, rc.top, rc.right, rc.bottom);

		SelectObject (hdc, hOldPen);
		SelectObject (hdc, hOldBrush);

		DeleteObject (hPen);
		DeleteObject (hBrush);

		COLORREF crOld = SetTextColor (hdc, GetSysColor (COLOR_WINDOWTEXT));
		DrawFaceText (hdc, ecDigital, nHeight / 8, szTime, nLen, &Size, true);
		SetTextColor (hdc, crOld);

	} else
		DrawFaceText (hdc, ecDigital, nHeight / 8, szTime, nLen, &Size);
#endif // KEYBOARD

}


void CClock::DrawWebAddress (HDC hdc, const RECT& rcFace)

{	
	int nHeight = rcFace.bottom - rcFace.top;
	m_Component [ecCyberspace].SetRect (rcFace.left,
		m_ptCentre.y + nHeight / 3, rcFace.right, rcFace.bottom);

	DrawFaceText (hdc, ecCyberspace, nHeight / 32,
		_T ("ClockFace (c) 2017 Dylan Harris"), 43);

	if (m_Component [ecCyberspace].IsShown ()) {

#if defined (_REG)
		USES_CONVERSION;

		LPSTR lpszText = OLE2A (m_bstrCaption.m_str);

		if (lpszText == NULL || lpszText [0] == 0)
			lpszText = "www.dylanharris.org";

#else // _REG
		LPSTR lpszText = "www.dylanharris.org";
#endif // _REG

		int y = m_ptCentre.y + (3 * nHeight) / 16;
		m_Component [ecCyberspace].SetRect (rcFace.left, y, rcFace.right, rcFace.bottom);

		SIZE sz;
		DrawFaceText (hdc, ecCyberspace, nHeight / 13, lpszText, strlen (lpszText), &sz);

		m_Component [ecCyberspace].SetRect (
			m_ptCentre.x - sz.cx / 2, y, m_ptCentre.x + sz.cx / 2, y + sz.cy);

	}

}



void CClock::DrawAMPM (HDC hdc, const RECT& rcFace,
	const int nHour, const int nMinute, const int nSecond,
	const HPEN hPenBack, const HBRUSH hBrushBack)

{
	long ltime;
	time (&ltime);
	struct tm *pTime = gmtime (&ltime);

	pTime -> tm_sec = nSecond;
    pTime -> tm_min = nMinute;
	pTime -> tm_hour = nHour;

	int nX = m_ptCentre.x + (rcFace.right - rcFace.left) / 6;
	SIZE sz;

	if (m_Component [ecAMPM].m_Drawn == CClockComponent::edDrawn) {

		TCHAR szTime [16];
		size_t nLen = _tcsftime (szTime, 64, _T ("%p"), pTime);

		GetTextExtentPoint32 (hdc, szTime, nLen, &sz);

		m_Component [ecAMPM].SetRect (
			nX - sz.cx / 2, 
			m_ptCentre.y - sz.cy / 2,
			nX + sz.cx / 2,
			m_ptCentre.y + sz.cy / 2);
		
		COLORREF crOld = SetTextColor (hdc, m_Component [ecAMPM].GetColour (GetState (ecAMPM)));

		if (crOld != CLR_INVALID) {

			DrawText (hdc, szTime, static_cast <int> (nLen), &m_Component [ecAMPM].m_rcIci,
				DT_CENTER | DT_TOP | DT_NOCLIP | DT_NOPREFIX | DT_SINGLELINE);
			SetTextColor (hdc, crOld);

		}

	} else
	if (m_Component [ecAMPM].m_Drawn == CClockComponent::edFancy) {

		sz.cx = (rcFace.right - rcFace.left) / 12;
		sz.cy = (rcFace.bottom - rcFace.top) / 12;

		bool bPM = nHour > 11;

		m_Component [ecAMPM].SetRect (
			nX - sz.cx / 2, 
			m_ptCentre.y - sz.cy / 2,
			nX + sz.cx / 2,
			m_ptCentre.y + sz.cy / 2);
		
		HPEN hPen = m_Component [ecAMPM].CreatePen (PS_SOLID, GetState (ecAMPM));

		if (hPen != NULL) {

			if (SelectObject (hdc, hPen) != NULL) {

				DrawHalfMoon (hdc, m_Component [ecAMPM].m_rcIci, bPM,
					m_Component [(! IsWindowEnabled ()) ? ecBackground : ecAMPM].GetColour (CClockComponent::esDisabled), hBrushBack);
				DrawHalfMoon (hdc, m_Component [ecAMPM].m_rcIci, ! bPM,
					m_Component [ecAMPM].GetColour (GetState (ecAMPM)), hBrushBack);
				SelectObject (hdc, hPenBack);

			}

			DeleteObject (hPen);

		}

	}

}


void CClock::DrawHalfMoon (HDC hdc, const RECT rc, const bool bRHS, const COLORREF cr, const HBRUSH hBrushBack)

{
	HBRUSH hBrush = CreateSolidBrush (cr);

	if (hBrush != NULL) {

		if (SelectObject (hdc, hBrush) != NULL) {

			int nX = (rc.right + rc.left) / 2;
			Chord (hdc, rc.left, rc.top, rc.right, rc.bottom,
				nX,	bRHS ? rc.bottom : rc.top,
				nX, bRHS ? rc.top : rc.bottom);
			SelectObject (hdc, hBrushBack);

		}

		DeleteObject (hBrush);

	}

}



void CClock::DrawHands (HDC hdc, const RECT& rcFace, const HPEN hPenBack,
	int nHour, const int nMinute, const int nSecond)

{
	if (m_Component [ecSecond].IsShown ())
		DrawHand (hdc, ecSecond, rcFace, nSecond, hPenBack);

	if (m_Component [ecHour].IsShown ()) {

		int nHourPos = (nHour > 11) ? (nHour - 12) : nHour;
		nHourPos *= 5;
		nHourPos += nMinute / 12;
		DrawHand (hdc, ecHour, rcFace, nHourPos, hPenBack);

	}

	if (m_Component [ecMinute].IsShown ())
		DrawHand (hdc, ecMinute, rcFace, nMinute, hPenBack);

}





void CClock::DrawHand (HDC hdc, const eComponent c,	const RECT& rcFace, 
	const int nPosition, const HPEN hPenBack)

{
	POINT ptTo = PointInCircle (nPosition, rcFace, m_Component [c].m_dbSize);

	if (m_Component [c].m_Drawn == CClockComponent::edFancy) {

		HPEN hPen = CreatePen (PS_SOLID,
			m_Component [c].m_nThickness + 2,
			m_Component [ecBackground].GetColour (GetState (c)));

		if (hPen != NULL) {
	
			if (SelectObject (hdc, hPen) != NULL) {
				POINT ptFrom = PointInCircle (nPosition, rcFace, m_Component [ecCentre].m_dbSize * 1.1);
				MoveToEx (hdc, ptFrom.x, ptFrom.y, NULL);
				LineTo (hdc, ptTo.x, ptTo.y);
				SelectObject (hdc, hPenBack);

			}

			DeleteObject (hPen);

		}

	}

	HPEN hPen = m_Component [c].CreatePen (PS_SOLID,GetState (c));

	if (hPen == NULL)
		return;
	
	if (SelectObject (hdc, hPen) != NULL) {

		MoveToEx (hdc, m_ptCentre.x, m_ptCentre.y, NULL);
		LineTo (hdc, ptTo.x, ptTo.y);
		m_Component [c].SetGravityRect (ptTo, m_szGravityWell);
		SelectObject (hdc, hPenBack);

	}

	DeleteObject (hPen);

}



void CClock::DrawFocus (HDC hdc, const RECT& rcFace, HPEN hPenBack)

{
	LOGBRUSH lb;

	lb.lbStyle = BS_SOLID;
	lb.lbColor = m_Component [ecFocus].GetColour ();
	lb.lbHatch = 0;

	HPEN hPenCircle = ExtCreatePen (PS_COSMETIC | PS_ALTERNATE,
		m_Component [ecFocus].m_nThickness, &lb, 0, NULL);

	if (hPenCircle == NULL)
		return;

	if (SelectObject (hdc, hPenCircle) != NULL) {

		int nWidthOffset = static_cast <int> (
			static_cast <double> (rcFace.right - rcFace.left) * m_Component [ecFocus].m_dbSize);
		int nHeightOffset = static_cast <int> (
			static_cast <double> (rcFace.bottom - rcFace.top) * m_Component [ecFocus].m_dbSize);

		Ellipse (hdc,	rcFace.left + nWidthOffset,
						rcFace.top + nHeightOffset,
						rcFace.right - nWidthOffset,
						rcFace.bottom - nHeightOffset);		
		SelectObject (hdc, hPenBack);

	}

	DeleteObject (hPenCircle);

}







STDMETHODIMP CClock::get_Colour(int nProperty, int nState, OLE_COLOR * pVal)

{
	_ASSERTE (! IsBadWritePtr (pVal, sizeof (OLE_COLOR)));

	if (! IsBadWritePtr (pVal, sizeof (OLE_COLOR)))
		if (nProperty > ecOutside && nProperty < ecMaximum)
			switch (nState) {

				case CClockComponent::esNormal :
					*pVal = (OLE_COLOR) m_Component [nProperty].m_crNormal;
					break;

				case CClockComponent::esSelected :
					*pVal = (OLE_COLOR) m_Component [nProperty].m_crSelected;
					break;

				case CClockComponent::esDisabled :
					*pVal = (OLE_COLOR) m_Component [nProperty].m_crDisabled;
					break;

			}

	return S_OK;
}

STDMETHODIMP CClock::put_Colour(int nProperty, int nState, OLE_COLOR newVal)
{
	if (nProperty > ecOutside && nProperty < ecMaximum)
		switch (nState) {

			case CClockComponent::esNormal :
				OleTranslateColor (newVal, NULL, &m_Component [nProperty].m_crNormal);
				break;

			case CClockComponent::esSelected :
				OleTranslateColor (newVal, NULL, &m_Component [nProperty].m_crSelected);
				break;

			case CClockComponent::esDisabled :
				OleTranslateColor (newVal, NULL, &m_Component [nProperty].m_crDisabled);
				break;

			case CClockComponent::esAll :
				OleTranslateColor (newVal, NULL, &m_Component [nProperty].m_crNormal);
				OleTranslateColor (newVal, NULL, &m_Component [nProperty].m_crSelected);
				OleTranslateColor (newVal, NULL, &m_Component [nProperty].m_crDisabled);
				break;

			default :
				return S_OK;

		}

	PerhapsFireViewChange ();
	return S_OK;
}

STDMETHODIMP CClock::get_Thickness(int nProperty, short * pVal)
{
	_ASSERTE (! IsBadWritePtr (pVal, sizeof (short)));

	if (! IsBadWritePtr (pVal, sizeof (short)))
		if (nProperty > ecOutside && nProperty < ecMaximum)
			*pVal = m_Component [nProperty].m_nThickness;

	return S_OK;
}

STDMETHODIMP CClock::put_Thickness(int nProperty, short newVal)
{
	if (nProperty > ecOutside && nProperty < ecMaximum)	{

		m_Component [nProperty].m_nThickness = newVal;
		PerhapsFireViewChange ();

	}
	
	return S_OK;
}

STDMETHODIMP CClock::get_Drawn(int nProperty, short * pVal)
{
	_ASSERTE (! IsBadWritePtr (pVal, sizeof (short)));

	if (! IsBadWritePtr (pVal, sizeof (short)))
		if (nProperty > ecOutside && nProperty < ecMaximum)
			*pVal = m_Component [nProperty].m_Drawn;

	return S_OK;
}

STDMETHODIMP CClock::put_Drawn(int nProperty, short newVal)
{
	if (nProperty > ecOutside && nProperty < ecMaximum)	{

		m_Component [nProperty].m_Drawn = static_cast <CClockComponent::eDrawn> (newVal);
		PerhapsFireViewChange ();

	}
	
	return S_OK;
}

STDMETHODIMP CClock::get_Size(int nProperty, double * pVal)
{
	_ASSERTE (! IsBadWritePtr (pVal, sizeof (double)));

	if (! IsBadWritePtr (pVal, sizeof (double)))
		if (nProperty > ecOutside && nProperty < ecMaximum)
			*pVal = m_Component [nProperty].m_dbSize;

	return S_OK;
}

STDMETHODIMP CClock::put_Size(int nProperty, double newVal)
{
	if (nProperty > ecOutside && nProperty < ecMaximum)	{

		m_Component [nProperty].m_dbSize = newVal;
		PerhapsFireViewChange ();

	}
	
	return S_OK;
}





STDMETHODIMP CClock::About()
{
	CAbout dlg;

	dlg.DoModal ();
	return S_OK;
}



LRESULT CClock::OnKeyDown (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)

{
	if (! IsWindowEnabled ())
		return TRUE;

	SetControlFocus (TRUE);

	switch (wParam) {

		case VK_SHIFT :
			m_bShift = true;
			break;

		case VK_CONTROL :
			m_bControl = true;
			break;

#if defined (KEYBOARD)
		case VK_ESCAPE :
			FinishTextEdit (true);
			break;
#endif // KEYBOARD

		case 'A' :
			{	int nHour, nMinute, nSecond;

				TimeFromDate (nHour, nMinute, nSecond);

				if (nHour >= 12)	{

					SetTime (nHour - 12, nMinute, nSecond);
					PerhapsFireViewChange ();
					Fire_Update (nHour, nMinute, nSecond);
					m_bChanged = true;

				}

			}
								
			break;

		case 'C' :
		case 'X' :
			if (m_bControl)
				Copy ();
								
			break;

		case 'H' :
			IncrementTime (euHour, m_bShift, lParam & 0xFFFF);
			break;

		case 'M' :
			IncrementTime (euMinute, m_bShift, lParam & 0xFFFF);
			break;

		case 'P' :
			{	int nHour, nMinute, nSecond;

				TimeFromDate (nHour, nMinute, nSecond);

				if (nHour < 12)	{

					SetTime (nHour + 12, nMinute, nSecond);
					PerhapsFireViewChange ();
					Fire_Update (nHour, nMinute, nSecond);
					m_bChanged = true;

				}

			}
								
			break;

		case 'S' :
			IncrementTime (euSecond, m_bShift, lParam & 0xFFFF);
			break;

		case 'V' :
			if (m_bControl)
				Paste ();
								
			break;

		case 'Z' :
			if (m_bControl)
#if defined (KEYBOARD)
				if (m_bEditText)
					FinishTextEdit (true);

				else
#endif // KEYBOARD
					Undo ();
								
			break;

		case VK_LEFT :
#if defined (KEYBOARD)
			EditTimeText (etLeft);
			break;
#endif // KEYBOARD
		case VK_UP :
		case VK_SUBTRACT :
			if (m_bShift)
				IncrementTime (euSecond, true, lParam & 0xFFFF);

			else
			if (m_bControl)
				IncrementTime (euHour, true, lParam & 0xFFFF);

			else
				IncrementTime (euMinute, true, lParam & 0xFFFF);

			break;

		case VK_RIGHT :
#if defined (KEYBOARD)
			EditTimeText (etRight);
			break;
#endif // KEYBOARD
		case VK_DOWN :
		case VK_ADD :
			if (m_bShift)
				IncrementTime (euSecond, false, lParam & 0xFFFF);

			else
			if (m_bControl)
				IncrementTime (euHour, false, lParam & 0xFFFF);

			else
				IncrementTime (euMinute, false, lParam & 0xFFFF);

			break;

		case VK_PRIOR :
		case VK_DIVIDE :
			if (m_bShift)
				IncrementTime (euMinute, false, lParam & 0xFFFF);

			else
			if (m_bControl)
				IncrementTime (euSecond, false, lParam & 0xFFFF);

			else
				IncrementTime (euHour, false, lParam & 0xFFFF);

			break;

		case VK_NEXT :
		case VK_MULTIPLY :
			if (m_bShift)
				IncrementTime (euMinute, true, lParam & 0xFFFF);

			else
			if (m_bControl)
				IncrementTime (euSecond, true, lParam & 0xFFFF);

			else
				IncrementTime (euHour, true, lParam & 0xFFFF);

			break;

#if defined (KEYBOARD)
		case '0' :
		case VK_NUMPAD0 :
			EditTimeText (etKey, '0');
			break;

		case '1' :
		case VK_NUMPAD1 :
			EditTimeText (etKey, '1');
			break;

		case '2' :
		case VK_NUMPAD2 :
			EditTimeText (etKey, '2');
			break;

		case '3' :
		case VK_NUMPAD3 :
			EditTimeText (etKey, '3');
			break;

		case '4' :
		case VK_NUMPAD4 :
			EditTimeText (etKey, '4');
			break;

		case '5' :
		case VK_NUMPAD5 :
			EditTimeText (etKey, '5');
			break;

		case '6' :
		case VK_NUMPAD6 :
			EditTimeText (etKey, '6');
			break;

		case '7' :
		case VK_NUMPAD7 :
			EditTimeText (etKey, '7');
			break;

		case '8' :
		case VK_NUMPAD8 :
			EditTimeText (etKey, '8');
			break;

		case '9' :
		case VK_NUMPAD9 :
			EditTimeText (etKey, '9');
			break;

		case VK_END :
			EditTimeText (etEnd);
			break;

		case VK_HOME :
			EditTimeText (etHome);
			break;
#endif // KEYBOARD

		default :
			return TRUE;

	}

	return FALSE;

}



			
LRESULT CClock::OnKeyUp (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)

{
	if (! IsWindowEnabled ())
		return TRUE;

	switch (wParam) {

		case VK_SHIFT :
			m_bShift = false;
			break;

		case VK_CONTROL :
			m_bControl = false;
			break;

		default :
			return TRUE;

	}

	return FALSE;

}
			

#if defined (KEYBOARD)
void CClock::EditTimeText (const eEdit Edit, const char ch)

{
	if (m_bEditText) {

		int nLen = strlen (m_szEditedText);

		switch (Edit) {

			case etLeft :
				if (m_nCursor > 0)
					m_nCursor--;

				break;

			case etRight :
				if (m_nCursor < nLen)
					m_nCursor = nLen;

				break;

			case etHome :
				m_nCursor = 0;
				break;

			case etEnd :
				m_nCursor = nLen;
				break;

			case etKey :
				if (nLen >= g_nMaxEditText - 1) {

					::MessageBeep (MB_ICONEXCLAMATION);
					break;

				}

				for (int i = m_nCursor; i <= nLen; i++)
					m_szEditedText [i+1] = m_szEditedText [i];

				m_szEditedText [m_nCursor++] = ch;
				break;
								 

		}

	} else {

		m_bEditText = true;
		int nH, nM, nS;
		TimeFromDate (nH, nM, nS);
		int nLen = TimeToText (m_szEditedText, g_nMaxEditText, nH, nM, nS);
		m_nSelStart = m_nSelEnd = m_nCursor = 0;

		switch (Edit) {

			case etDown :
				m_nSelEnd = -1;
				break;
				
			case etLeft :
			case etHome :
				break;

			case etRight :
			case etEnd :
				m_nCursor = nLen;
				break;

			case etKey :
				m_szEditedText [0] = ch;
				m_szEditedText [1] = 0;
				m_nCursor = 1;
				break;

		}

	}
	
	PerhapsFireViewChange ();
}


void CClock::FinishTextEdit (const bool bCancel)

{
	if (! m_bEditText)
		return;

	if (! bCancel) {

		int nHour, nMinute, nSecond, nH, nM, nS;
		TCHAR tch [g_nMaxEditText];
		TimeFromDate (nH, nM, nS);
		int i;

		switch (sscanf (m_szEditedText, "%d:%d:%d%s", &nHour, &nMinute, &nSecond, tch)) {

			case 0 :
				m_bEditText = false;
				PerhapsFireViewChange ();
				return;

			case 1 :
				nMinute = nM;

			case 2 :
				nSecond = nS;
				break;

			case 4 :

				if (! m_szAM [0] || ! m_szPM [0])
					break;

				for (i = 0; i < g_nMaxEditText; i++) {

					if (tch [i] == m_szAM [0]) {

						if (nHour > 11)
							nHour -= 12;

						break;

					}

					if (tch [i] == m_szPM [0]) {
						
						if (nHour < 12)
							nHour += 12;

						break;

					}
					
				}

				break;

		}

		if (nHour >= 0 && nHour < 24)
			if (nMinute >= 0 && nMinute < 60)
				if (nSecond >= 0 && nMinute < 60)
					SetTime (nHour, nMinute, nSecond);

	}

	m_bEditText = false;
	PerhapsFireViewChange ();
}
#endif // KEYBOARD


STDMETHODIMP CClock::Undo()

{	int nHour, nMinute, nSecond;

	m_Time = m_UndoTime;
	TimeFromDate (nHour, nMinute, nSecond);
	PerhapsFireViewChange ();
	Fire_Update (nHour, nMinute, nSecond);
	m_bChanged = true;
	return S_OK;
}

STDMETHODIMP CClock::CanUndo(int *bCanUndo)
{
	_ASSERTE (! IsBadWritePtr (bCanUndo, sizeof (int)));

	if (! IsBadWritePtr (bCanUndo, sizeof (int)))
		*bCanUndo = (m_Time != m_UndoTime);

	return S_OK;
}

STDMETHODIMP CClock::get_DigitalFormat(BSTR * pVal)
{
	_ASSERTE (! IsBadWritePtr (pVal, sizeof (BSTR)));

#if defined (NO_BSTR)
	USES_CONVERSION;

	if (! IsBadWritePtr (pVal, sizeof (BSTR))) {

		CComBSTR x (m_szDigitalFormat);
		*pVal = x.Copy ();
//		*pVal = T2OLE (m_szDigitalFormat);

	}

#else // NO_BSTR
	if (! IsBadWritePtr (pVal, sizeof (BSTR)))
		*pVal = m_bstrDigitalFormat;
#endif // NO_BSTR

	return S_OK;
}

STDMETHODIMP CClock::put_DigitalFormat(BSTR newVal)
{
#if defined (NO_BSTR)
	USES_CONVERSION;

	_tcsncpy (m_szDigitalFormat, OLE2T (newVal), g_nMaxEditText - 1);
#else // NO_BSTR
	m_bstrDigitalFormat = newVal;
#endif // NO_BSTR
	return S_OK;
}




void CClock::CheckColours ()

{	int i = 0;

	if (m_clrOldBackColor != m_clrBackColor) {

		for (i = 0; i < ecMaximum; i++)
			if (g_rgnDefault [i] == g_nBack)
				m_Component [i].SetColour (m_clrBackColor);

		m_clrOldBackColor =	m_clrBackColor;

	}

	if (m_clrOldBorderColor != m_clrBorderColor) {

		for (i = 0; i < ecMaximum; i++)
			if (g_rgnDefault [i] == g_nBorder)
				m_Component [i].SetColour (m_clrBorderColor);

		m_clrOldBorderColor =  m_clrBorderColor;

	}

	if (m_clrOldFillColor != m_clrFillColor) {

		for (i = 0; i < ecMaximum; i++)
			if (g_rgnDefault [i] == g_nFill)
				m_Component [i].SetColour (m_clrFillColor);

		m_clrOldFillColor =	m_clrFillColor;

	}

	if (m_clrOldForeColor != m_clrForeColor) {

		for (i = 0; i < ecMaximum; i++)
			if (g_rgnDefault [i] == g_nFore)
				m_Component [i].SetColour (m_clrForeColor);

		m_clrOldForeColor =	m_clrForeColor;

	}

	if (m_nOldBorderWidth != m_nBorderWidth) {

		m_Component [ecCircle].SetWidth (m_nBorderWidth);
		m_nOldBorderWidth = m_nBorderWidth;

	}

	if (m_nOldDrawMode != m_nDrawMode) {

		switch (m_nDrawMode) {

			case 1 : // vbBlackness
				m_dwDrawMode = BLACKNESS;
				break;

			case 2 : // vbNotMergePen
				m_dwDrawMode = MERGEPAINT;
				break;

			case 3 : // vbMaskNotPen
				m_dwDrawMode = NOTSRCERASE;
				break;
 
			case 4 : // vbNotCopyPen
				m_dwDrawMode = NOTSRCCOPY;
				break;
 
			case 5 : // vbMaskPenNot
 				m_dwDrawMode = PATPAINT;
				break;
 
			case 6 : // vbInvert
				m_dwDrawMode = DSTINVERT;
				break;
 
			case 7 : // vbXorPen
				m_dwDrawMode = SRCINVERT;
				break;
 
			case 8 : // vbNotMaskPen
				m_dwDrawMode = SRCERASE;
				break;
 
			case 9 : // vbMaskPen
				m_dwDrawMode = SRCAND;
				break;
 
			case 10 : // vbNotXorPen
				m_dwDrawMode = SRCINVERT;
				break;
 
			case 11 : // vbNop
				m_dwDrawMode = -1;
				break;
 
			case 12 : // vbMergeNotPen
				m_dwDrawMode = PATINVERT;
				break;
 
//			case 13 : // vbCopyPen
//				m_dwDrawMode = SRCCOPY;
//				break;
 
			case 14 : // vbMergePenNot
				m_dwDrawMode = PATPAINT;
				break;
 
			case 15 : // vbMergePen
				m_dwDrawMode = MERGECOPY;
				break;
 
			case 16 : // vbWhiteness
				m_dwDrawMode = WHITENESS;
				break;

			default :
				m_dwDrawMode = SRCCOPY;
				break;

		}

		m_nOldDrawMode = m_nDrawMode;

	}

}



const int g_nMax = 64;
					  



STDMETHODIMP CClock::Copy()
{
	if (OpenClipboard ()) {

		if (::EmptyClipboard ()) {

			HANDLE hText = ::GlobalAlloc (GMEM_MOVEABLE | GMEM_DDESHARE, g_nMax);
			LPTSTR lpTime = static_cast <LPTSTR> (GlobalLock (hText));

			if (lpTime == NULL) {

				GlobalFree (hText);
				return S_OK;

			}

			int nHour, nMinute, nSecond;
			TimeFromDate (nHour, nMinute, nSecond);
			TimeToText (lpTime, g_nMax,	nHour, nMinute, nSecond, true);
			::GlobalUnlock (hText);

#if defined (_UNICODE)
			::SetClipboardData (CF_UNICODETEXT, hText);
#else // _UNICODE
			::SetClipboardData (CF_TEXT, hText);
#endif // _UNICODE
		}

		::CloseClipboard ();

	}

	return S_OK;
}




bool CClock::CheckPaste (const bool bDoIt)

{
	bool bProcessed = false;

	if (OpenClipboard ()) {
		
#if defined (_UNICODE)
		if (::IsClipboardFormatAvailable (CF_UNICODETEXT)) {

			HGLOBAL hData = ::GetClipboardData (CF_UNICODETEXT);
#else // _UNICODE
		if (::IsClipboardFormatAvailable (CF_TEXT)) {

			HGLOBAL hData = ::GetClipboardData (CF_TEXT);
#endif // _UNICODE

			if (hData != NULL) {
	
				LPTSTR lpData = static_cast <LPTSTR> (::GlobalLock (hData));
				
				if (lpData != NULL) {
		
					int nHour, nMinute, nSecond;

					if (_stscanf (lpData, _T ("%d:%d:%d"), &nHour, &nMinute, &nSecond) == 3)
						bProcessed =	(nHour >= 0) && (nHour < 24) &&
										(nMinute >= 0) && (nMinute < 60) &&
										(nSecond >= 0) && (nSecond < 60);

					::GlobalUnlock (hData);
	
					if (bProcessed && bDoIt)
						SetTime (nHour, nMinute, nSecond);

				}

			}

		}

		::CloseClipboard ();

	}

	return bProcessed;

}




STDMETHODIMP CClock::Paste()

{
	CheckPaste (true);
	return S_OK;
}


STDMETHODIMP CClock::CanPaste(int * bCanPaste)
{
	_ASSERTE (! IsBadWritePtr (bCanPaste, sizeof (int)));

	if (! IsBadWritePtr (bCanPaste, sizeof (int)))
		*bCanPaste = !! CheckPaste (false);

	return S_OK;
}

STDMETHODIMP CClock::get_LegendIPAddress(BSTR * pVal)
{
	_ASSERTE (! IsBadWritePtr (pVal, sizeof (BSTR)));

#if defined (NO_BSTR)
	USES_CONVERSION;

	if (! IsBadWritePtr (pVal, sizeof (BSTR))) {

		CComBSTR x (m_szIPAddress);
		*pVal = x.Copy ();
//		*pVal = T2OLE (m_szIPAddress);

	}

#else // NO_BSTR
	if (! IsBadWritePtr (pVal, sizeof (BSTR)))
		*pVal = m_bstrIPAddress;
#endif // NO_BSTR

	return S_OK;
}

STDMETHODIMP CClock::put_LegendIPAddress(BSTR newVal)
{
#if defined (_REG)
#if defined (NO_BSTR)
	USES_CONVERSION;

	_tcsncpy (m_szIPAddress, OLE2T (newVal), _MAX_PATH);
#else // NO_BSTR
	m_bstrIPAddress = newVal;
#endif // NO_BSTR
#else // _REG
	MessageBox (_T ("You must register the ClockFace control to be able to set the LegendIPAddress property."),
				_T ("ClockFace"), MB_ICONEXCLAMATION); 
#endif // _REG

	return S_OK;
}



void CClock::LegendLink ()

{
	USES_CONVERSION;

#if defined (NO_BSTR)
	if (! m_szIPAddress [0])
#else // NO_BSTR
	LPSTR lpszIPAddress = OLE2A (m_bstrIPAddress.m_str);

	if (lpszIPAddress == NULL || lpszIPAddress [0] == 0)
#endif // NO_BSTR
		return;

	HCURSOR hOld = SetCursor (LoadCursor (NULL, IDC_WAIT));

	if (! InetIsOffline (0))
//		if (InternetCheckConnection (lpszIPAddress, FLAG_ICC_FORCE_CONNECTION, 0))
			HlinkSimpleNavigateToString (
#if defined (NO_BSTR)
				T2W (m_szIPAddress), NULL, NULL, 
#else
				OLE2W (m_bstrIPAddress.m_str), NULL, NULL, 
#endif
				GetUnknown(), NULL, NULL, HLNF_OPENINNEWWINDOW, 0);

	SetCursor (hOld != NULL ? hOld : LoadCursor (NULL, IDC_ARROW));

}

 



STDMETHODIMP CClock::GetVersion(int * pMajor, int * pMinor, int * pRelease, int * pBuild)
{
	if (! IsBadWritePtr (pMajor, sizeof (int)))
		*pMajor = g_nMajor;

	if (! IsBadWritePtr (pMinor, sizeof (int)))
		*pMinor = g_nMinor;

	if (! IsBadWritePtr (pRelease, sizeof (int)))
		*pRelease = g_nRelease;

	if (! IsBadWritePtr (pBuild, sizeof (int)))
		*pBuild = g_nBuild;

	return S_OK;
}




void CClock::Duplicate (const CClock& c)

{
	_ASSERTE (FALSE);

	for (int i = 0; i < ecMaximum; i++)
		m_Component [i] = c.m_Component [i];
		
	for (i = 0; i < etMaximum; i++)
		m_rgnTimer [i] = c.m_rgnTimer [i];	

	m_bCapture = c.m_bCapture;
	m_bDown = c.m_bDown;
	m_bFocus = c.m_bFocus;
	m_bControl = c.m_bControl;
	m_bShift = c.m_bShift;
	m_bChanged = c.m_bChanged;
	m_szGravityWell = c.m_szGravityWell;
	m_ecDown = c.m_ecDown;	// eComponent
	m_nWinderTimer = c.m_nWinderTimer;
	m_ptCentre.x = c.m_ptCentre.x;
	m_ptCentre.y = c.m_ptCentre.y;
	m_Time = c.m_Time;
	m_UndoTime = c.m_UndoTime;
#if defined (NO_BSTR)
	_tcsncpy (m_szDigitalFormat, c.m_szDigitalFormat, g_nMaxEditText);
	_tcsncpy (m_szIPAddress, c.m_szIPAddress, _MAX_PATH);
#else // NO_BSTR
	m_bstrDigitalFormat = c.m_bstrDigitalFormat;
	m_bstrIPAddress = c.m_bstrIPAddress;
#endif // NO_BSTR
#if defined (KEYBOARD)
//	TCHAR		m_szAM [g_nID], m_szPM [g_nID];
//	int			m_nSelStart, m_nSelEnd, m_nCursor;
//	bool		m_bEditText;
#endif // KEYBOARD

	m_clrOldBackColor = c.m_clrOldBackColor; // OLE_COLOR
	m_clrOldBorderColor = c.m_clrOldBorderColor;
	m_clrOldFillColor = c.m_clrOldFillColor;
	m_clrOldForeColor = c.m_clrOldForeColor;

	m_nOldBorderWidth = c.m_nOldBorderWidth;
	m_dwDrawMode = c.m_dwDrawMode;
	m_nOldDrawMode = c.m_nOldDrawMode;

#if defined (KEYBOARD)
//	TCHAR		m_szEditedText [g_nMaxEditText];
#endif // KEYBOARD

	m_pFont = c.m_pFont; // CComPtr<IFontDisp>
	m_pMouseIcon = c.m_pMouseIcon; // CComPtr<IPictureDisp> 
	m_pPicture = c.m_pPicture;
	m_clrBackColor = c.m_clrBackColor;	// OLE_COLOR 
	m_clrBorderColor = c.m_clrBorderColor;
	m_clrFillColor = c.m_clrFillColor;
	m_clrForeColor = c.m_clrForeColor;
	m_bstrText = c.m_bstrText;
	m_bstrCaption = c.m_bstrCaption;
	m_bValid = c.m_bValid;
	m_bTabStop = c.m_bTabStop;
	m_bBorderVisible = c.m_bBorderVisible;
	m_nBackStyle = c.m_nBackStyle;
	m_nBorderStyle = c.m_nBorderStyle;
	m_nBorderWidth = c.m_nBorderWidth;
	m_nDrawMode = c.m_nDrawMode;
	m_nDrawStyle = c.m_nDrawStyle;
	m_nDrawWidth = c.m_nDrawWidth;
	m_nFillStyle = c.m_nFillStyle;
	m_nAppearance = c.m_nAppearance;
	m_nMousePointer = c.m_nMousePointer;

	m_bDrawn = false;
	m_bEnable = c.m_bEnable;

}




