// Clock.h : Declaration of the CClock

#ifndef __CLOCK_H_
#define __CLOCK_H_

#define NO_BSTR

#include <math.h>
#include "resource.h"       // main symbols
#include "CPClockFace.h"


const int g_nMajor = 1;
const int g_nMinor = 0;
const int g_nRelease = 2;
const int g_nBuild = 10;

// #define KEYBOARD
// #define LICENSE_DEBUG

const long g_nMinDate = -657434L;
const long g_nMaxDate =	2958465L;
const double g_dbHalfSecond = 1.0/172800.0;
const double g_dbPi = 3.1415926535;

const int g_nMaxEditText = 32;
const int g_nID = 4;



class CClockComponent

{
public:
	typedef enum {

		edHidden, edDrawn, edFancy

	} eDrawn;

	typedef enum {

		esNormal, esSelected, esDisabled, esAll

	} eState;

	COLORREF	m_crNormal, m_crSelected, m_crDisabled;
	RECT		m_rcIci;
	int			m_nThickness;
	eDrawn		m_Drawn;
	double		m_dbSize;

private:
	void Construct ()
		{
			m_crNormal = RGB (128, 128, 128);
			m_crSelected = RGB (0, 0, 0);
			m_crDisabled = RGB (255, 255, 255);
			SetRectEmpty (&m_rcIci);
			m_nThickness = 0;
			m_dbSize = 0.0;
			m_Drawn = edHidden;
		}

	void Duplicate (const CClockComponent& cc)
		{
			m_crNormal = cc.m_crNormal;
			m_crSelected = cc.m_crSelected;
			m_crDisabled = cc.m_crDisabled;
			CopyRect (&m_rcIci, &cc.m_rcIci);
			m_nThickness = cc.m_nThickness;
			m_Drawn = cc.m_Drawn;
			m_dbSize = cc.m_dbSize;
		}

public:
	void Set (	const COLORREF crNormal, 
				const COLORREF crSelected, 
				const COLORREF crDisabled,
				const RECT& rcIci,
				const int nThickness,
				const eDrawn Drawn,
				const double dbSize)
		{
			m_crNormal = crNormal;
			m_crSelected = crSelected;
			m_crDisabled = crDisabled;
			CopyRect (&m_rcIci, &rcIci);
			m_nThickness = nThickness;
			m_Drawn = Drawn;
			m_dbSize = dbSize;
		}

	CClockComponent ()
		{
			Construct ();
		}

	CClockComponent (const CClockComponent& cc)
		{
			Duplicate (cc);
		}

	CClockComponent (	const COLORREF crNormal, 
						const COLORREF crSelected, 
						const COLORREF crDisabled,
						const RECT& rcIci,
						const int nThickness,
						const eDrawn Drawn,
						const double dbSize)
		{
			Set (crNormal, crSelected, crDisabled, rcIci, nThickness, Drawn, dbSize);
		}

	CClockComponent& operator = (const CClockComponent& cc)
		{
			Duplicate (cc);
			return *this;
		}

	void SetRect ()
		{
			SetRectEmpty (&m_rcIci);
		}

	void SetRect (const RECT& rc)
		{
			CopyRect (&m_rcIci, &rc);
		}

	void SetRect (const int left, const int top, const int right, const int bottom)
		{
			::SetRect (&m_rcIci, left, top, right, bottom);
		}

	bool IsEmpty () const
		{
			return !! IsRectEmpty (&m_rcIci);
		}

	bool PtInRect (const POINT pt)
		{
			return ! IsEmpty () && ::PtInRect (&m_rcIci, pt);
		}

	bool IsShown () const
		{
			return m_Drawn != edHidden;
		}

	COLORREF GetColour (const unsigned eState = esNormal) const
		{
			if (eState == esNormal)
				return m_crNormal;

			if (eState == esSelected)
				return m_crSelected;

			return m_crDisabled;
		}

	void SetColour (const COLORREF cr, const unsigned eState = esAll)
		{
			if (eState == esNormal)
				m_crNormal = cr;

			else
			if (eState == esSelected)
				m_crSelected = cr;

			else
			if (eState == esDisabled)
				m_crDisabled = cr;

			else
				m_crNormal = m_crSelected = m_crDisabled = cr;
		}

	void SetGravityRect (const POINT pt, const SIZE szGravity)
		{
			m_rcIci.left = pt.x - szGravity.cx;
			m_rcIci.right = pt.x + szGravity.cx;
			m_rcIci.top = pt.y - szGravity.cy;
			m_rcIci.bottom = pt.y + szGravity.cy;
		}

	HPEN CreatePen (const int uStyle, const unsigned eState = esNormal)
		{
			return ::CreatePen (uStyle, m_nThickness, GetColour (eState));
		}

	HBRUSH CreateSolidBrush (const unsigned eState = esNormal)
		{
			return ::CreateSolidBrush (GetColour (eState));
		}

	void SetWidth (const int nWidth = 0)
		{
			m_nThickness = nWidth;
		}
};

/*
const TCHAR tchLicense [] = _T ("ClockFace ActiveX Control, (c) 2017 Dylan Harris, https://dylanharris.org/"); 

class CClockLicense
{
protected:
	static BOOL VerifyLicenseKey (BSTR bstr)
		{
#if defined (LICENSE_DEBUG)
			::MessageBox (NULL, "VerifyLicenseKey", "ClockFace", MB_OK | MB_ICONINFORMATION);
#endif // DEBUG
			USES_CONVERSION;
			return TRUE;
//			return ! _tcscmp (OLE2T (bstr), tchLicense);
		}

	static BOOL GetLicenseKey (DWORD dwReserved, BSTR* pBstr) 
		{
#if defined (LICENSE_DEBUG)
			::MessageBox (NULL, "GetLicenseKey", "ClockFace", MB_OK | MB_ICONINFORMATION);
#endif // DEBUG
			USES_CONVERSION;
			*pBstr = SysAllocString (T2OLE (tchLicense)); 
			return TRUE;
		}

	static BOOL IsLicenseValid ()
		{  
#if defined (LICENSE_DEBUG)
			::MessageBox (NULL, "IsLicenseValid", "ClockFace", MB_OK | MB_ICONINFORMATION);
#endif // DEBUG
			return TRUE;		
		}

};
*/


/////////////////////////////////////////////////////////////////////////////
// CClock
class ATL_NO_VTABLE CClock : 
	public CComObjectRootEx<CComSingleThreadModel>,
	public CComCoClass<CClock, &CLSID_Clock>,
	public CComControl<CClock>,
	public CStockPropImpl<CClock, IClock, &IID_IClock, &LIBID_CLOCKFACELib>,
	public IProvideClassInfo2Impl<&CLSID_Clock, &DIID__ClockEvents, &LIBID_CLOCKFACELib>,
	public IPersistStreamInitImpl<CClock>,
	public IPersistStorageImpl<CClock>,
	public IQuickActivateImpl<CClock>,
	public IOleControlImpl<CClock>,
	public IOleObjectImpl<CClock>,
	public IOleInPlaceActiveObjectImpl<CClock>,
	public IViewObjectExImpl<CClock>,
	public IOleInPlaceObjectWindowlessImpl<CClock>,
	public IDataObjectImpl<CClock>,
	public ISpecifyPropertyPagesImpl<CClock>,
    public CProxy_ClockEvents<CClock>,
    public IConnectionPointContainerImpl<CClock>,
	public IObjectSafetyImpl<CClock>
{
public:
	typedef enum {

		ecOutside,
		ecHour, ecMinute, ecSecond,
		ecWindUp, ecWindDown, ecAMPM, ecCircle, 
		ecQuarterGrad, ecFiveGrad, ecSingleGrad,
		ecWinderArm, ecCentre, ecCyberspace, ecDigital,
		ecFocus, ecBackground,
		ecMaximum

	} eComponent;

	typedef enum {

		etNone,
		etWindStart, etWindRepeat,
		etMaximum

	} eTimer;

	typedef enum {

		euHour, euMinute, euSecond

	} eUnit;

#if defined (KEYBOARD)
	typedef enum {

		etDown, etLeft, etRight, etKey, etEnd, etHome

	} eEdit;
#endif // KEYBOARD

private:
	CClockComponent	m_Component [ecMaximum];
	int	m_rgnTimer [etMaximum];

	bool		m_bCapture, m_bDown, m_bFocus, m_bControl,
				m_bShift, m_bChanged, m_bEnable;
	SIZE		m_szGravityWell;
	eComponent	m_ecDown;
	int			m_nWinderTimer;
	POINT		m_ptCentre;
	DATE		m_Time, m_UndoTime;
#if defined (NO_BSTR)
	TCHAR		m_szDigitalFormat [g_nMaxEditText];
	TCHAR		m_szIPAddress [_MAX_PATH];
#else // NO_BSTR
	CComBSTR	m_bstrDigitalFormat;
	CComBSTR	m_bstrIPAddress;
#endif // NO_BSTR
#if defined (KEYBOARD)
	TCHAR		m_szAM [g_nID], m_szPM [g_nID];
	int			m_nSelStart, m_nSelEnd, m_nCursor;
	bool		m_bEditText;
#endif // KEYBOARD

	OLE_COLOR	m_clrOldBackColor;
	OLE_COLOR	m_clrOldBorderColor;
	OLE_COLOR	m_clrOldFillColor;
	OLE_COLOR	m_clrOldForeColor;

	int			m_nOldBorderWidth;
	DWORD		m_dwDrawMode;
	long		m_nOldDrawMode;
	bool		m_bDrawn;

#if defined (KEYBOARD)
	TCHAR		m_szEditedText [g_nMaxEditText];
#endif // KEYBOARD

	eComponent GetHotSpot (const POINT pt);
	HCURSOR SetAppropriateCursor (const int nMinute);
	HCURSOR SetMouseCursor (const eComponent h, const POINT pt);
	void TimeFromDate (int& nHour, int& nMinute, int& nSecond);
	void SwapAMPM ();
	void ApplyWinder ();
	void ZapTimers ();
	bool DragTime (const POINT pt);
	int DragHand (const POINT pt, const int nMinute);

	void Uncapture ()
		{
			if (m_bCapture) {
				ReleaseCapture ();
				m_bCapture = false;
			}
		}

	static bool InWindow (const eComponent h)
		{
			return (h >= ecHour && h <= ecDigital);
//			return (h >= ecHour && h <= ecClient);
		}

	static bool InActiveArea (const eComponent h)
		{
			return (h >= ecHour && h <= ecAMPM);
		}

	bool IsWindowEnabled ()
		{
			return (!! m_bEnable);
		}

	CClockComponent::eState GetState (const eComponent c)
		{
			if (! m_bEnable)
				return CClockComponent::esDisabled;

			if (InActiveArea (c) && c == m_ecDown)
				return CClockComponent::esSelected;

			return CClockComponent::esNormal;
		}

	POINT PointInCircle (const int nMinute, const RECT& rc, const double dbOffset = 1.0)
		{	POINT ptMinute;
			double dbMinute = g_dbPi * static_cast <double> (nMinute) / 30.0;
			int nWidth = rc.right - rc.left;
			int nHeight = rc.bottom - rc.top;
			ptMinute.x = m_ptCentre.x +
				static_cast <int> (sin (dbMinute) * 
					static_cast <double> (nWidth) * 0.5 * dbOffset);
			ptMinute.y = m_ptCentre.y -
				static_cast <int> (cos (dbMinute) * 
					static_cast <double> (nHeight) * 0.5 * dbOffset);
			return ptMinute;
		}

	HDC CreateGDIObjects (ATL_DRAWINFO& di, HBITMAP& hBit, HBITMAP& hOldBitmap,
		HPEN& hPenBack, HPEN& hOldPen, HBRUSH& hBrushBack, HBRUSH& hOldBrush,
		const SIZE szBounds);
	void DeleteGDIObjects (	HDC hdc, const HBITMAP hBit, const HBITMAP hOldBitmap,
		const HPEN hPenBack, const HPEN hOldPen, 
		const HBRUSH hBrushBack, const HBRUSH hOldBrush);
	void DrawOuterCircle (HDC hdc, const RECT& rcFace, HPEN hPenBack);
	void DrawFocus (HDC hdc, const RECT& rcFace, HPEN hPenBack);
	void CalculateFacePosition (const RECT& rcBounds, RECT& rcFace, SIZE& szWinder);
	void DrawCentre (HDC hdc, const RECT& rcFace, const HPEN hPenBack, const HBRUSH hBrushBack);
	void DrawRect (HDC hdc, const eComponent c, const HPEN hPenBack, const HBRUSH hBrushBack);
	void DrawWinder (HDC hdc, const RECT& rcFace, const RECT rcBounds,
		const SIZE szWinder, const HPEN hPenBack, const HBRUSH hBrushBack);
	void DrawGraduations (HDC hdc, const RECT& rcFace, const HPEN hPenBack);
	void DrawTimeText (HDC hdc, const RECT& rcFace, 
		const int nHour, const int nMinute, const int nSecond);
	void DrawAMPM (HDC hdc, const RECT& rcFace,
		const int nHour, const int nMinute, const int nSecond,
		const HPEN hPenBack, const HBRUSH hBrushBack);
	void DrawHalfMoon (HDC hdc, const RECT rc, const bool bRHS, const COLORREF cr, const HBRUSH hBrushBack);
	void DrawHand (HDC hdc, const eComponent c,	const RECT& rcFace, 
		const int nPosition, const HPEN hPenBack);
	void DrawHands (HDC hdc, const RECT& rcFace, const HPEN hPenBack,
		int nHour, const int nMinute, const int nSecond);
	void DrawWebAddress (HDC hdc, const RECT& rcFace);
	void DrawBackground (HDC hdc, const RECT& rcFace, HBRUSH hBrushBack);
	HFONT CreateHFont (HDC hDC, const int nHeight);
	void DrawFaceText (HDC hDC, const eComponent e, const int nHeight,
#if ! defined (KEYBOARD)
		LPCTSTR lpszText, const int nLength, LPSIZE = NULL);
#else // KEYBOARD
		LPCTSTR lpszText, const int nLength, LPSIZE lpSize = NULL, const bool bSpecial = false);
#endif // KEYBOARD
	void IncrementTime (const eUnit Unit, const bool bBack, const int nCount = 1);
	void CheckColours ();
#if defined (KEYBOARD)
	void EditTimeText (const eEdit Edit, const char ch = 0);
	void FinishTextEdit (const bool bCancel = false);
#endif // KEYBOARD
	int TimeToText (TCHAR szTime [], const int nMax,
		const int nHour, const int nMinute, const int nSecond, const bool bDefault = false);
	bool CheckPaste (const bool bDoIt);
	void LegendLink ();

	void Construct ();
	void Duplicate (const CClock& c);

	bool SetTimeSimply (const int nHour, const int nMinute, const int nSecond);

	void PerhapsFireViewChange ()
		{
			if (m_bDrawn)
				FireViewChange ();
		}

public:
	CClock()
		{
			Construct ();
		}

	CClock (const CClock& c)
		{
			Duplicate (c);
		}

//DECLARE_CLASSFACTORY2(CClockLicense)
DECLARE_CLASSFACTORY()
DECLARE_REGISTRY_RESOURCEID(IDR_CLOCK)

BEGIN_COM_MAP(CClock)
	COM_INTERFACE_ENTRY(IClock)
	COM_INTERFACE_ENTRY(IDispatch)
	COM_INTERFACE_ENTRY_IMPL(IViewObjectEx)
	COM_INTERFACE_ENTRY_IMPL_IID(IID_IViewObject2, IViewObjectEx)
	COM_INTERFACE_ENTRY_IMPL_IID(IID_IViewObject, IViewObjectEx)
	COM_INTERFACE_ENTRY_IMPL(IOleInPlaceObjectWindowless)
	COM_INTERFACE_ENTRY_IMPL_IID(IID_IOleInPlaceObject, IOleInPlaceObjectWindowless)
	COM_INTERFACE_ENTRY_IMPL_IID(IID_IOleWindow, IOleInPlaceObjectWindowless)
	COM_INTERFACE_ENTRY_IMPL(IOleInPlaceActiveObject)
	COM_INTERFACE_ENTRY_IMPL(IOleControl)
	COM_INTERFACE_ENTRY_IMPL(IOleObject)
	COM_INTERFACE_ENTRY_IMPL(IQuickActivate)
	COM_INTERFACE_ENTRY_IMPL(IPersistStorage)
	COM_INTERFACE_ENTRY_IMPL(IPersistStreamInit)
	COM_INTERFACE_ENTRY_IMPL(ISpecifyPropertyPages)
	COM_INTERFACE_ENTRY_IMPL(IDataObject)
	COM_INTERFACE_ENTRY(IProvideClassInfo)
	COM_INTERFACE_ENTRY(IProvideClassInfo2)
	COM_INTERFACE_ENTRY_IMPL(IConnectionPointContainer)
	COM_INTERFACE_ENTRY_IMPL(IObjectSafety)
END_COM_MAP()

BEGIN_CONNECTION_POINT_MAP(CClock)
    CONNECTION_POINT_ENTRY(DIID__ClockEvents)
END_CONNECTION_POINT_MAP()


BEGIN_PROPERTY_MAP(CClock)
	// Example entries
	// PROP_ENTRY("Property Description", dispid, clsid)
	PROP_ENTRY("Hands", 1, CLSID_ClockProp)
	PROP_ENTRY("Face", 1, CLSID_Circle)
	PROP_ENTRY("Other", 1, CLSID_Element)
	PROP_ENTRY("Time", 1, CLSID_Time)
    PROP_PAGE (CLSID_StockColorPage)
	PROP_PAGE (CLSID_StockFontPage)
END_PROPERTY_MAP()


BEGIN_MSG_MAP(CClock)
	MESSAGE_HANDLER(WM_PAINT, OnPaint)
	MESSAGE_HANDLER(WM_SETFOCUS, OnSetFocus)
	MESSAGE_HANDLER(WM_KILLFOCUS, OnKillFocus)
    MESSAGE_HANDLER(WM_LBUTTONDOWN, OnLButtonDown)
    MESSAGE_HANDLER(WM_LBUTTONUP, OnLButtonUp)
    MESSAGE_HANDLER(WM_MOUSEMOVE, OnMouseMove)
    MESSAGE_HANDLER(WM_CLOSE, OnClose)
    MESSAGE_HANDLER(WM_ERASEBKGND, OnErase)
    MESSAGE_HANDLER(WM_TIMER, OnTimer)
    MESSAGE_HANDLER(WM_ENABLE, OnEnable)
    MESSAGE_HANDLER(WM_SETFOCUS, OnSetFocus)
    MESSAGE_HANDLER(WM_KILLFOCUS, OnKillFocus)
    MESSAGE_HANDLER(WM_MBUTTONDOWN, OnMButtonDown)
    MESSAGE_HANDLER(WM_MBUTTONUP, OnMButtonUp)
    MESSAGE_HANDLER(WM_RBUTTONDOWN, OnRButtonDown)
    MESSAGE_HANDLER(WM_RBUTTONUP, OnRButtonUp)
    MESSAGE_HANDLER(WM_KEYDOWN, OnKeyDown)
    MESSAGE_HANDLER(WM_KEYUP, OnKeyUp)
    MESSAGE_HANDLER(WM_CHAR, OnChar)
	MESSAGE_HANDLER(WM_GETDLGCODE, OnSortOutArrows)
    MESSAGE_HANDLER(WM_SHOWWINDOW, OnShowWindow)
END_MSG_MAP()


// IViewObjectEx
	STDMETHOD(GetViewStatus)(DWORD* pdwStatus)
	{
		ATLTRACE(_T("IViewObjectExImpl::GetViewStatus\n"));
		_ASSERTE (! IsBadWritePtr (pdwStatus, sizeof (DWORD)));
		*pdwStatus = VIEWSTATUS_SOLIDBKGND | VIEWSTATUS_OPAQUE;
		return S_OK;
	}

// IClock
public:
	STDMETHOD(GetVersion)(/*[out]*/ int* pMajor, /*[out]*/ int* pMinor, /*[out]*/ int* pRelease, /*[out]*/ int* pBuild);
	STDMETHOD(get_LegendIPAddress)(/*[out, retval]*/ BSTR *pVal);
	STDMETHOD(put_LegendIPAddress)(/*[in]*/ BSTR newVal);
	STDMETHOD(CanPaste)(int* bCanPaste);
	STDMETHOD(Paste)();
	STDMETHOD(Copy)();
	STDMETHOD(get_DigitalFormat)(/*[out, retval]*/ BSTR *pVal);
	STDMETHOD(put_DigitalFormat)(/*[in]*/ BSTR newVal);
	STDMETHOD(CanUndo)(/*[out, retval]*/ int *bCanUndo);
	STDMETHOD(Undo)();
	STDMETHOD(About)();
	STDMETHOD(get_Size)(int nProperty, /*[out, retval]*/ double *pVal);
	STDMETHOD(put_Size)(int nProperty, /*[in]*/ double newVal);
	STDMETHOD(get_Drawn)(int nProperty, /*[out, retval]*/ short *pVal);
	STDMETHOD(put_Drawn)(int nProperty, /*[in]*/ short newVal);
	STDMETHOD(get_Thickness)(int nProperty, /*[out, retval]*/ short *pVal);
	STDMETHOD(put_Thickness)(int nProperty, /*[in]*/ short newVal);
	STDMETHOD(get_Colour)(int nProperty, int nState, /*[out, retval]*/ OLE_COLOR *pVal);
	STDMETHOD(put_Colour)(int nProperty, int nState, /*[in]*/ OLE_COLOR newVal);
	STDMETHOD(get_GravityWellY)(/*[out, retval]*/ long *pVal);
	STDMETHOD(put_GravityWellY)(/*[in]*/ long newVal);
	STDMETHOD(get_GravityWellX)(/*[out, retval]*/ long *pVal);
	STDMETHOD(put_GravityWellX)(/*[in]*/ long newVal);
	STDMETHOD(SetCurrentTime)();
	STDMETHOD(SetTime)(int nHour, int nMinute, int nSecond);
	STDMETHOD(get_Time)(/*[out, retval]*/ DATE *pVal);
	STDMETHOD(put_Time)(/*[in]*/ DATE newVal);

	HRESULT OnDraw(ATL_DRAWINFO& di);
	LRESULT OnMouseMove (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled);
 	LRESULT OnLButtonDown (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled);
	LRESULT OnLButtonUp (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled);
	LRESULT OnTimer (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled);
 	LRESULT OnKeyDown (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled);
	LRESULT OnKeyUp (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled);

	LRESULT OnSortOutArrows (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
		{
			return DLGC_WANTARROWS | DLGC_WANTCHARS;
		}

	LRESULT OnChar (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
		{
			return 0;
		}

	LRESULT OnShowWindow (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
		{
			m_bDrawn = !! wParam;
			return 1;
		}

	LRESULT OnEnable (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
		{
			if ((! m_bEnable) != (! wParam)) {
				
				m_bEnable = !! wParam;
				PerhapsFireViewChange ();

			}

			return TRUE;
		}

	LRESULT OnSetFocus (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
		{
			if (! m_bFocus) {

				m_bFocus = true;
#if defined (KEYBOARD)
				m_bEditText = false;
#endif // KEYBOARD

				PerhapsFireViewChange ();

			}

			return TRUE;
		}

	LRESULT OnKillFocus (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
		{
			if (m_bFocus) {

				m_UndoTime = m_Time;
				m_bFocus = false;
#if defined (KEYBOARD)
				FinishTextEdit ();
#endif // KEYBOARD
				PerhapsFireViewChange ();

			}

			return TRUE;
		}

	LRESULT OnErase (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
		{
			return TRUE;
		}

	LRESULT OnClose (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
		{
#if defined (KEYBOARD)
			FinishTextEdit ();
#endif // KEYBOARD
			Uncapture ();
			ZapTimers ();
			bHandled = false;
			return TRUE;
		}

 	LRESULT OnMButtonDown (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
		{
			SetControlFocus (TRUE);
			Fire_MButtonDown (LOWORD (lParam), HIWORD (lParam));
			return TRUE;
		}

	LRESULT OnMButtonUp (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
		{
			Fire_MButtonUp (LOWORD (lParam), HIWORD (lParam));
			return TRUE;
		}

 	LRESULT OnRButtonDown (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
		{
			SetControlFocus (TRUE);
			Fire_RButtonDown (LOWORD (lParam), HIWORD (lParam));
			return TRUE;
		}

	LRESULT OnRButtonUp (UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
		{
			Fire_RButtonUp (LOWORD (lParam), HIWORD (lParam));
			return TRUE;
		}


	CComPtr<IFontDisp> m_pFont;
	CComPtr<IPictureDisp> m_pMouseIcon;
	CComPtr<IPictureDisp> m_pPicture;
	OLE_COLOR m_clrBackColor;
	OLE_COLOR m_clrBorderColor;
	OLE_COLOR m_clrFillColor;
	OLE_COLOR m_clrForeColor;
	CComBSTR m_bstrText;
	CComBSTR m_bstrCaption;
	BOOL m_bValid;
	BOOL m_bTabStop;
	BOOL m_bBorderVisible;
	long m_nBackStyle;
	long m_nBorderStyle;
	long m_nBorderWidth;
	long m_nDrawMode;
	long m_nDrawStyle;
	long m_nDrawWidth;
	long m_nFillStyle;
	long m_nAppearance;
	long m_nMousePointer;

};

#endif //__CLOCK_H_
