/*
 *	name		ALARM.C
 *	author		Anthony Hay, Digital Research (UK) Ltd.,
 *	date created	5-3-86
 *	last modified	2-6-86
 *	purpose		To demonstrate DOS Plus background processing
 *			capability.
 *
 *	One copy of ALARM executes in background and maintains a list of up to 
 *	32  alarm entries read initially from a disk file.   A foreground copy 
 *	of  ALARM allows alarms to be added (set) or removed  (canceled)  from 
 *	this  disk file.   The ALARM running in background is terminated and a 
 *	new one initialised with the modified ALARM list.   Additionally,  the 
 *	program can be removed with the ALARMs preserved or cancelled.  When a 
 *	ALARM  date/time  arrives the specified message is  displayed  on  the 
 *	status line until the user acknoledges the alarm. 
 *
 *
 *	ALARM                       
 *                Date      Time      Message 
 *                --------------------------------------------------
 *                dd/mm/yy  hh:mm     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
 *
 *                ALARM ON  or  ALARM OFF  accordingly.
 *          
 *	ALARM  date time message
 *	ALARM  date time /C
 *                 "ALARM entry not found"
 *	ALARM /ON      
 *                 "No alarms set"
 *                 "Can't find alarm file ALARM.DAT"
 *	ALARM /OFF    
 *	ALARM /T       
 *
 *
 * 	history
 *	29-4-86	Ant	Fixed zero length ALARM.DAT file problem.
 *			Alarm is now activated when the user enters an alarm.
 *	1-5-86	Ant	Added alarm_xx messages.
 *	2-6-86  Ant	Added GEMALARM.ACC interface.
 *
 * 		Copyright (c) 1986 Digital Research Inc.
 */
 
#include <stdio.h>
#include <bdosf.h>
UWORD	__cpmrv;		/* BDOS error code */

char	copyright[] = { "Copyright (c) 1986 Digital Research Inc." };

/* messages */
extern char	alarm_yes_char;
extern char	alarm_00[];
extern char	alarm_01[];
extern char	alarm_02[];
extern char	alarm_03[];
extern char	alarm_04[];
extern char	alarm_05[];
extern char	alarm_07[];
extern char	alarm_08[];
extern char	alarm_09[];
extern char	alarm_06[];
extern char	alarm_10[];
extern char	alarm_11[];
extern char	alarm_12[];
extern char	alarm_13[];
extern char	alarm_14[];
extern char	alarm_15[];
extern char	alarm_16[];
extern char	alarm_17[];

/* command line options searched for */
extern char	alarm_x1[];
extern char	alarm_x2[];
extern char	alarm_x3[];

/* external procedures */
extern	VOID 	isbrother();
extern	VOID 	killbrother();
extern	char 	phys_drive();
extern	VOID 	get_sem();
extern	VOID 	release_sem();
extern 	UWORD	my_ds();

/* external GEMALARM.ACC communication area */
extern	VOID	put_drive();
extern	VOID	put_cmd();
extern	VOID	put_msg_off();
extern	char	get_drive();
extern	char	get_cmd();


/* forward ref */ 
BYTE	*deblank();
BYTE 	*index();
char	*get_item();
char	*get_eoln();
char	*rol_to_ascii();
char	*num();
char	uc();


struct	CPMTIME
{
UWORD	day;			/* days since 1/1/78 */
BYTE	hour;			/* BCD */
BYTE	min;			/* BCD */
BYTE	secs;			/* BCD */
};  

struct	MYTIME
{
UWORD	day;			/* days since 1/1/78 */
BYTE	hour;			/* cardinal */
BYTE	min;			/* cardinal */
};  


/* The following refer to the gem_cmd variable for */
/* comminication with GEMALARM */
#define	NO_OP		0
#define	SHOW_MSG	1
#define	RE_READ		2
#define	READ_ERROR	3


#define NUM_ALARMS	32
#define	BUF_SIZE	1500
#define	MSG_SIZE	43	/* max number of characters in any */
				/* message + 3 (the first two bytes are */
				/* used for max and actual string lengths, */
				/* the last byte is for null termination) */

#define	MAX_ARGC	3	/* max number of command line parameters */
				/* that will be extracter + 1 */


char		buf[BUF_SIZE];			/* ALARM.DAT data read into here */
struct MYTIME	alarm[NUM_ALARMS];	 	/* list of alarm dates */
char		msgs[NUM_ALARMS][MSG_SIZE];	/* list of message strings */
UWORD		dseg;				/* program data segment */


_main( s )
char	*s;
{
    main( s );
    bdos( P_TERM, 0 );
}

main ( s )
char	*s;
{
char		*argv;
struct MYTIME	a;

    /* first check that this is a suitable operasting system for ALARM */
    os_check();

    /* check that ALARM has not been "execed" from a DOS program */
    if ( exec_check() )
    {					/* warn user and terminate if it has */
	printf( alarm_16 );
	printf( alarm_17 );
	bdos( C_RAWIO, 0xFD );		/* wait for key */
	bdos( P_TERM, 0 );
    }

    get_date( &a );			/* by default a is the current time */
    dseg = my_ds();
    bdos( F_ERRMODE, 0x00FF );		/* set Return Error Mode */

    /* ALARM expects to see the file ALARM.DAT to exist in the root */
    /* of the drive from which ALARM was loaded, L: */
    put_drive( phys_drive( 'L'-'A' ) );
    put_cmd( NO_OP );

printf( "ds: %x\r\n", dseg );

    if ( ! get_arg( &s, &argv ) )
    {					/* display list of set alarms */
	load_alarms();			/* read alarms from ALARM.DAT */
     	show_alarms();			/* display table of alarms */
	show_onoff();			/* tell user the alarm status */
    }
    	
    					/* is it one of the /options ? */
    else if ( !xstrcmp( argv, alarm_x1 ) )	/* "/ON" */
    {
    	load_alarms();			/* read alarms from ALARM.DAT */
	killbrother();			/* kill off any existing alarms */
	alarm_on();			/* enter background mode */
    }	
	
    else if ( !xstrcmp( argv, alarm_x2 ) )	/* "/OFF" */
	killbrother();
	
    else if ( !xstrcmp( argv, alarm_x3 ) )	/* "/T" */
    {
	printf( alarm_00 );	/* "Confirm remove all alarms (Y/N)? " */
	if ( uc( bdos( C_READ ) ) == alarm_yes_char )
	{
	    killbrother();
	    buf[0] = '\0';
	    if ( write_file( buf ) )
	    bad_exit( 4 );		/* Unable to write to file */
	}
    }

    else if ( check_time( argv, &a.hour, &a.min ) )
					/* a.day already set to today */
	do_alarm( &a, s );
	
    else if ( a.day = check_date( argv ) )
    {
					/* the second parameter is the time */
	if ( ! get_arg( &s, &argv ) )
	    bad_exit( 1 );

	if ( ! check_time( argv, &a.hour, &a.min ) )
	    bad_exit( 1 );

	do_alarm( &a, s );
    }

    else 
	bad_exit( 1 );		/* "Bad Option." */
}



VOID	do_alarm( a, s )
struct MYTIME	*a;
char		*s;
{
    load_alarms();		/* read alarms from ALARM.DAT */

				/* the last parameter is '/c' or a message */
    if ( *s && !xstrncmp( s, "/C", 2 ) ) 
    {
	if ( del_alarm( a ) )
	    bad_exit( 2 );	/* "Alarm entry not found." */
    }
    else
    {
	if ( add_alarm( a, s ) )
	    bad_exit( 3 );	/* "Alarm table full." */
    }

#if 0
    if ( isbrother() )		/* if the alarm is running in back g. */
    {
	killbrother();		/* replace it with this one */
	alarm_on();
    }
#else
    /* Steve requested that alarm be activated when a valid alarm has been */
    /* set rather than waiting for the user to type ALARM /ON. (29-4-86)   */
    if ( isbrother() )		/* if the alarm is running in back g. */
	killbrother();		/* replace it with this one */
    alarm_on();
#endif
}




/* To get one space terminated argument from s */
int	get_arg( s, argv )
char	**s;
char	**argv;
{
					/* skip white space */
    	while ( **s && ((**s == ' ') || (**s == '\t')) )
	    (*s)++;
	    				/* exit for if nothing on command line */
	if ( ! **s )
	    return 0;
					/* record address of this parameter */
	*argv = *s;
					/* make parameter lower case */
	while ( **s && (**s != ' ') && (**s != '\t') )
	    (*s)++;
	    				/* null terminate parameter */
	if ( **s )
	{
	    **s = '\0';
	    (*s)++;
	}
	return 1;
    }





int	del_alarm( a )
struct MYTIME	*a;
{
int	i;

    for (i = 0; i < NUM_ALARMS; i++)
    	if ( equal( a, &alarm[i] ) )
	{
	    alarm[i].day = 0;			/* turn off alarm */
	    if ( save_alarms() )		/* update alarm file */
	        bad_exit( 4 );			/* "Unable to write to ALARM.DAT */
	    return 0;				/* return success */
	}

    return 1;					/* return failure */
}


int	add_alarm( a, s )
struct MYTIME	*a;
char		*s;
{
int	i;
char	*m;

    for (i = 0; i < NUM_ALARMS; i++)
	if (!alarm[i].day)
	    break;

    if (alarm[i].day)
    	return 1;				/* table full */

    alarm[i].day = a->day;
    alarm[i].hour = a->hour;
    alarm[i].min = a->min;

    m = msgs[i];
    while ( *m++ = *s++ )
    	;

    if ( save_alarms() )			/* update alarm file */
        bad_exit( 4 );			/* "Unable to write to ALARM.DAT */

    return 0;					/* success */
}




VOID	show_onoff()
{
    if ( isbrother() )			/* see if there is a process with */
    	printf( alarm_01 ); /*"ALARM ON\r\n"*/	/* the same name as me already */
    else				/* running in the background */
    	printf( alarm_02 );		/* "ALARM OFF\r\n" */
}






/* To read the file ALARM.DAT into buf and initialise the list of alarm */
/* dates and corresponding messages. Returns 0 on success, 1 if can't */
/* read file in which case the alarm data structure is nulled. */
int	load_alarms()
{
static struct MYTIME	now;
int			i;

					/* clear alarms and messages */
    for ( i = 0; i < NUM_ALARMS; i++ )
    {
	alarm[i].day = 0;
	msgs[i][0] = '\0';
    }

					/* read data file into buf */
    if ( get_alarm_dates( buf, BUF_SIZE ) )
        return 1;
					/* extract dates and messages */ 
    get_dates();

    					/* remove any alarms that have expired */
    get_date( &now );
    for (i = 0; i < NUM_ALARMS; i++ )	
    	if ( alarm[i].day && before( &alarm[i], &now ) )
	    alarm[i].day = 0;		/* turn entry off */

    return 0;
}


BOOLEAN	save_alarms()
{
int	i;
char	*m, *p;

    p = buf;

    for (i = 0; i < NUM_ALARMS; i++)
    	if ( alarm[i].day )
	{
	    p = rol_to_ascii( alarm[i].day, p );
	    *p++ = ' ';
	    p = num( alarm[i].hour, p, 2 );
	    *p++ = ':';
	    p = num( alarm[i].min, p, 2 );
	    *p++ = ' ';
	    m = msgs[i];
	    while ( *m )
	    	*p++ = *m++;
	    *p++ = '\r';
	    *p++ = '\n';
	}
 
    *p = '\0';

    return write_file( buf );
}




VOID	alarm_on()
{
static struct MYTIME	now;
int			i;

    bdos( C_DETACH, 0 );

#if 0
    while ( dates_remain( alarm, NUM_ALARMS ) )
#else	
    FOREVER	    	      /* for the benifit of GEMALARM, stay resident */
#endif
    {
	bdos( P_DELAY, 50 );		/* wait 1 sec (assume 50 ticks/sec) */
	re_read_chk();			/* re-read ALARM.DAT if requested */
    	get_date( &now );		/* get current system date and time */

	for (i = 0; i < NUM_ALARMS; i++ )	
	{
	    if ( alarm[i].day && before( &alarm[i], &now ) )
	    {
		display_msg( msgs[i] );
		alarm[i].day = 0;	/* turn entry off */
	    }
	}
	release_sem( dseg );		/* allow GEMALARM back in */
    } 
     
					/* on exit from this loop all alarms */
					/* have expired - nothing more to do */
    bdos( P_TERM, 0 );
}



VOID	display_msg( msg )
char	*msg;
{
    /* we always try to display the message on the status line */
    status_msg( msg );

    /* Signal that we have a message to display. If GEMALARM.ACC is */
    /* present it will pick this signal up and put up an alert box. */
    put_msg_off( msg );			/* pass offset of message */
    put_cmd( SHOW_MSG );
}


VOID	re_read_chk()
{
    FOREVER
    {
    	get_sem( dseg );		/* lock out GEMALARM.ACC */

	if ( get_cmd() != RE_READ )
	    return;

	if ( load_alarms() )
	    put_cmd( READ_ERROR );
	else
	    put_cmd( NO_OP );

	release_sem( dseg );		/* allow GEMALARM back in */
	bdos( P_DELAY, 50 );		/* wait a sec */
    }
}





VOID	show_alarms()
{
int	i;
char	s[11];
BOOLEAN	swaped;

    if ( ! dates_remain( alarm, NUM_ALARMS ) )
    {
	printf( alarm_03 );		/* "No alarms set\r\n" */
	return;
    }

				/* sort alarms */
    do
    {
	swaped = NO;
    	for ( i = 0; i < NUM_ALARMS-1; i++ )
				/* this will put all the null alarms to the fore */
	    if ( ! before( &alarm[i], &alarm[i+1] ) )
	    {
	    	swap( &alarm[i], &alarm[i+1] );
		swapm( msgs[i], msgs[i+1] );
		swaped = YES;
	    }
    } while ( swaped );



    printf( alarm_04 );	/* "Date         Time      Message\r\n" */
    printf( alarm_05 );	/* "---------------------------------------------------------\r\n" */

    for ( i = 0; i < NUM_ALARMS; i++ )
    {
    	if ( alarm[i].day )
	{
	    rol_to_ascii( alarm[i].day, s );
	    printf( "%s  ", s );
	    printf( "%2d:%02d", alarm[i].hour, alarm[i].min );
	    printf( "      %s\r\n", msgs[i] );
	}
    }

    printf( alarm_05 );
}






BOOLEAN	get_alarm_dates( b, size )
char	*b;				/* address of data buffer */
UWORD	size;				/* number of byte to read */
{
static char 	fcb[36] = "\0ALARM   DAT";
UWORD		i;
UWORD		ret;

    if ( cd_root() )			/* open ALARM.DAT in the root */
	return 1;

    fcb[0] = 'L' - 'A' + 1;		/* try to open ALARM.DAT */
    if ( bdos( F_OPEN, fcb ) )
      	return 1;
    
    bdos( F_DMAOFF, b );
    bdos( F_MULTISEC, size>>7 );

    bdos( F_SIZE, fcb );
    if ( ((fcb[34] << 8) + fcb[33]) == 0 )
    	return 1;

    ret = bdos( F_READ, fcb );
    bdos( F_CLOSE, fcb );

    if ( ret > 1 )
    	return 1;

					/* check all but the last byte */
    for ( i = 1; i < size; i++ )	/* in the buffer for EOF */	
    {
	if ( (*b == 0x00) || (*b == 0x1A) )
	    break;
    	b++;
    }
    *b = 0x1A;			/* there is always an EOF in the buffer */ 

    return 0;
}


int	write_file( b )
BYTE	*b;
{
static char 	fcb[36] = "\0ALARM   DAT";
UWORD		size;
char		*p;
int		ret;

    if ( cd_root() )			/* open ALARM.DAT in the root */
	return 1;

    fcb[0] = 'L' - 'A'  + 1;		/* write ALARM.DAT */
    bdos( F_DELETE, fcb );

    if ( bdos( F_MAKE, fcb ) )
    	return 1;

    p = b;
    size = 0;
    while ( *p )
    {
    	p++;
    	size++;
    }
    *p = 0x1A;

    bdos( F_DMAOFF, b );
    bdos( F_MULTISEC, (size>>7)+1 );
    ret = bdos( F_WRITE, fcb );
    bdos( F_CLOSE, fcb );

    if ( ret > 1 )
    	return 1;
}



VOID	cd_root()
{
static char	fcb[36] = "\0           ";

    fcb[0] = 'L' - 'A' + 1;		/* set the drive */
    fcb[0] |= 0x80;			/* root please */
    if ( bdos( F_CLOSE, fcb ) )
      	return 1;

    return 0;
}





int	dates_remain( d, num )
struct MYTIME	d[];
int		num;
{
int	i;
int	dr;

   dr = 0;
   for (i = 0; i < num; i++ )	
	if ( d[i].day )
	    dr++;

    return dr;
}



VOID	get_dates()
{
int		i, j;
char		*m, *p;
char		s[100];

    p = buf;
	
    for (i = 0;  i < NUM_ALARMS; i++)
    {
	p = get_item( p, s );
	if ( !(alarm[i].day = check_date( s )) )
	    break;

	p = get_item( p, s );
	if ( !check_time( s, &alarm[i].hour, &alarm[i].min ) )
	    break;

	p = get_eoln( p, msgs[i], MSG_SIZE ); 
    }

    for (j = i; j < NUM_ALARMS; j++)
        alarm[j].day = 0;
}


/* This proc will copy the next non-blank string from p into s */
char	*get_item( p, s )
char	*p, *s;
{
	p = deblank( p );
	while ( (*p != ' ') && 
		(*p != '\t') && 
		(*p != '\n') &&
		(*p != '\r') &&
		(*p != 0x1A) 
	      )
	    *s++ = *p++;
	*s = '\0';
	return p;
}


char	*get_eoln( p, m, max )
char	*m, *p;
int	max;
{
int	i;

    p = deblank( p );
					/* skip until EOLN or EOF */
    i = 1;
    while ( (*p != '\r') && (*p != '\n') && (*p != 0x1A) && (i++ < max) )
	*m++ = *p++;
    *m = '\0';
					/* skip rest of EOLN's */
    while ( ( (*p == '\r') || (*p == '\n') ) && (*p != 0x1A) )
	p++;

    return p;
}





BOOLEAN	before(d1, d2)
struct MYTIME	*d1, *d2;
{
    return (  (d1->day < d2->day) || 
    	     ((d1->day == d2->day) && (d1->hour < d2->hour)) ||
	     ((d1->day == d2->day) && (d1->hour == d2->hour) && (d1->min <= d2->min))
	   );
}


BOOLEAN	equal(d1, d2)
struct MYTIME	*d1, *d2;
{
    return ( (d1->day 	== d2->day) && 
    	     (d1->hour 	== d2->hour) &&
	     (d1->min 	== d2->min)
 	   );
}

VOID	swap(d1, d2)
struct MYTIME	*d1, *d2;
{
UWORD	w;
BYTE	b;

    w = d1->day;	d1->day = d2->day;	d2->day = w;
    b = d1->hour;	d1->hour = d2->hour;	d2->hour = b;
    b = d1->min;	d1->min = d2->min;	d2->min = b;
}


VOID	swapm( s1, s2 )
char	*s1, *s2;
{
char	temp[ MSG_SIZE ];

    xstrcpy( temp, s1 );
    xstrcpy( s1, s2 );
    xstrcpy( s2, temp );
}

VOID	xstrcpy( d, s )
char	*s, *d;
{
    while ( *d++ = *s++ )
    	;
}



VOID	get_date( d )
struct MYTIME *d;
{
static struct CPMTIME	date;

    bdos( T_GET, &date );
    d->day = date.day;					/* Rolander days */
    d->hour = (date.hour/16) * 10 + (date.hour%16);	/* BCD hours */
    d->min = (date.min/16) * 10 + (date.min%16); 	/* BCD minutes */
}    



 
/*
 *	Parse the string pointed to by s and check for a valid date
 *	specification. 
 */

BYTE month_count[] = {0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
BYTE date_sep[] = "/.-";

int	check_date(s)
BYTE 	*s;
{
	WORD	p1, p2, p3;		/* Input Parameters */
	WORD	day, month, year;
	WORD	days_this_month;

	deblank(s);			/* Remove spaces */

	if (!getdigit (&p1, &s) ||
	    !index (date_sep, *s++) ||
	    !getdigit (&p2, &s) ||
	    !index (date_sep, *s++) ||
	    !getdigit (&p3, &s))
	{
	    return 0;
	}

	switch (1) /*(country.dt_fmt)*/
	{
	 case 1:			/* European Format dd/mm/yy */
	    day = p1;
	    month = p2;
	    year = p3;
	    break;

	 case 2:			/* Japanese Format yy/mm/dd */
	    day = p3;
	    month = p2;
	    year = p1;
	    break;

	 default:			/* US Format mm/dd/yy */
	    day = p2;
	    month = p1;
	    year = p3;
	    break;
	}


	if (year >= 78 && year <= 99) 		/* (0.03) centuries */
	   year += 1900;			/* (0.03) are allowed */ 
	
	if (year < 1978 || year > 2099 ||	/* (0.03) check the Year */
	    month < 1 || month > 12)		/* Check the month */
	    return 0;

	days_this_month = month_count[ month ];
	if ( !(year % 4) && (month == 2) )
	    days_this_month++; 

	if ( (day < 1) || (day > days_this_month) )/* Range check the day */
	    return 0;

	return rolander(day, month, year);
}



int days_so_far[] = {0, 31, 59, 90, 120, 151, 181, 
			212, 243, 273, 304, 334, 365 };

int	rolander( d, m, y )
int	d, m, y;
{
int	x, days;

    x = y - 1978;		/* x = number of years since 1/1/78 */
    days = x * 365;		/* days = No. days since 1/1/78 ignoring leaps*/
    days += (x + 1) / 4;	/* add in number of leap years */
    
    days += days_so_far[ m-1 ];	/* add in days so far this year */
    if (!(y % 4) && (m > 2))	/* if leap year and February past... */
    	days += 1;		/* ...add in leap day */

    days += d;			/* add in days so far this month */
    
    return days;
}


char	*rol_to_ascii( days, s )
int	days;
char	*s;
{
int	days_this_year;
int	year, month, days_this_month;

    year = 1978;
    FOREVER
    {
	if ( year % 4 )
	    days_this_year = 365;
	else
	    days_this_year = 366;

    	if ( days <= days_this_year )
	    break;
	days -= days_this_year;

	year++;
    }

    month = 1;
    FOREVER
    {
    	days_this_month = month_count[ month ];
	if ( !(year % 4) && (month == 2) )
	    days_this_month++;

	if ( days <= days_this_month )
	    break;
	days -= days_this_month;

	month++;
    }

    s = num( days, s, 2 );
    *s++ = '/';
    s = num( month, s, 2 );
    *s++ = '/';
    s = num( year, s, 4 );
    *s = '\0';

    return s;
}


UWORD	ten[] = { 1, 10, 100, 1000 };

char	*num( d, s, l )
int	d;
char	*s;
int	l;
{
#if 0
    if ( d > 9 )
    	s = num(d/10, s );

    *s++ = (char)(d%10 + '0');
#endif

    while ( l-- )
    	*s++ = (char) ((d / ten[ l ]) % 10 ) + '0';

    return s;
}

/*
 *	Parse the string pointed to by s and check for a valid date
 *	specification. 
 */
BYTE	hour_sep[] = ":.";
BYTE	sec_sep[]  = ".,";


BOOLEAN check_time(s, h, m)		/* (0.03) changed extensively */
BYTE  	*s;				/* by jc to allow hour specifaction */
int	*h, *m;
{					/* alone */
WORD	hour, min, sec, hsec;

	min = sec = hsec = 0;		/* Seconds and Hundredths are optional */
					/* and default to zero when omitted */
	s = deblank (s);		/* Remove spaces */

	if (!getdigit (&hour, &s))
	    return NO;

	while(*s)				/* if more than HH */
	{
	    if( !index (hour_sep, *s++) ||	/* Check for Minute */
		!getdigit (&min, &s))
	    {
		return NO;
	    }

	    if(!*s)			/* Stop checking if HH:MM */
	        break;

	    if (!index (hour_sep, *s++) ||	/* Check for Seconds */
	        !getdigit (&sec, &s))
	    {
	        return NO;
	    }

	    if(!*s)			/* Stop checking if HH:MM:SS */
	        break;

	    if (!index (sec_sep, *s++) ||
		!getdigit (&hsec, &s))	/* check for hundreths */
	    {
		return NO;
	    }

	    if(!*s)			/* If this is not the end of the */
	        break;			/* string then return an error.	 */
	    else
	        return NO;
	}

	if (hour < 0  ||	/* Check the Hour */
	    hour > 23 ||
	    min < 0   ||	/* Check the minute */
	    min > 59  ||
	    sec < 0   ||	/* Check the Second */
	    sec > 59  ||
	    hsec < 0  ||	/* Check the Hundredths */
	    hsec > 99)
	{
	    return NO;
	}

	*h = hour;
	*m = min;
	return YES;
}



BOOLEAN check_num(s, min, max, value)
BYTE *s;	/* Input String */
WORD min, max;	/* Minimum and Maximum values */
WORD *value;	/* Value Input */
{
	WORD u;

	deblank(s);

	if(getdigit(&u, &s) == FALSE)
	    return FAILURE;

	if(*s)
	    return FAILURE;

	if(u < min || u > max)
	    return FAILURE;

	*value = u;
	return SUCCESS;
}

BOOLEAN getdigit(n, s)
WORD *n;			/* Pointer to the word number to save */
BYTE **s;			/* String to Process */
{

	*n = 0;				/* Zero the number */

	while(!isdigit(**s) && **s)	/* Skip all non digits */
	    (*s)++;

	if(**s) {
	    while(isdigit(**s)) {	/* Add all the digits in */
		*n = (**s - '0') + (*n * 10);
		(*s)++;
	    }
	    return TRUE;		/* and return success */
	}
	else
	    return FALSE;
}

BOOLEAN isdigit(b)
BYTE b;
{
	return (b >= '0' && b <= '9');
}


BYTE	*deblank (s)			/* scan off leading white space */
REG BYTE *s;				/* starting address of scan */
{
	while ((*s == ' ') ||		/* while there are more spaces */
	       (*s == '\t'))		/*   or tab characters ahead */
	    s ++;			/*   skip them */
	return (s);			/* return deblanked string */
}


char	uc( c )
char	c;
{
    if ( ('a' <= c) && (c <= 'z') )
    	return (c - 'a' + 'A');
    else
    	return c;
}



int	xstrcmp( s1, s2 )
char	*s1, *s2;
{
    while ( *s1 )
	if ( *s1++ != *s2++ )
	    return 1;
    return (*s1 != *s2);
}

int	xstrncmp( s1, s2, l )
char	*s1, *s2;
int	l;
{
    while ( l-- )
	if ( *s1++ != *s2++ )
	    return 1;

    return 0;
}




VOID	os_check()
{
UWORD	ret;

    bdos( S_BDOSVER );
    ret = __cpmrv;
    if ((((ret >> 8) & 0xfd) != 0x10) ||	/* if not DOS Plus ... */
	((ret & 0xff) < 0x41))			/* or version too old... */
    {
	bdos( C_WRITESTR, alarm_15 );
	bdos( P_TERM, 0 );
    }
}



VOID	bad_exit( e )
int	e;
{
    switch ( e )
    {
    case 1:
	printf( alarm_06 );
	printf( alarm_10 );
	printf( alarm_11 );
	printf( alarm_12 );
	printf( alarm_13 );
	printf( alarm_14 );
	break;
    case 2:
	printf( alarm_07 );	/* "Alarm entry not found.\r\n" */
	break;
    case 3:
	printf( alarm_08 );	/* "Alarm table full.\r\n" */
	break;
    case 4:
    	printf( alarm_09 );	/* "Unable to write to alarm file ALARM.DAT.\r\n" */
	break;
    }

    bdos( P_TERM, 0 );
}


/*********************   E N D   O F   A L A R M . C   ***********************/
