#pragma once

namespace lip {

	template <class T> class temp_buffer : private boost::noncopyable
	{
	private:
		T* ptr_;
		temp_buffer (); //lint !e1704
	public:
		explicit temp_buffer (const size_t count) : ptr_ (new T [count]) { _ASSERT (! ::IsBadWritePtr (ptr_, count * sizeof (T))); }
		explicit temp_buffer (const T* const p, const size_t count) : ptr_ (new T [count])
			{	_ASSERT (! ::IsBadReadPtr (p, count * sizeof (T)));
				_ASSERT (! ::IsBadWritePtr (ptr_, count * sizeof (T)));
				if (p != NULL) ::memcpy (ptr_, p, count * sizeof (T));  }
		~temp_buffer () throw () { clear (); }
		
		T* get () throw () { return ptr_; }
		const T* const get () const throw () { return ptr_; }
		T* data () throw () { return ptr_; }
		const T* const data () const throw () { return ptr_; }
		void clear () throw () { delete [] ptr_; ptr_ = NULL; }	// presumes delete never throws
		void swap (temp_buffer& other) throw () { const T* tmp = ptr_; ptr_ = other.ptr_; other.ptr_ = tmp; }
	};
	
	
	
	template <class T> struct temp_string : public temp_buffer <T>
	{
		explicit temp_string (const size_t count) : temp_buffer (count + 1) { }
	};
	
	
		
	
	template <class T> class secure_buffer
	{
	private:
		T* ptr_;
		unsigned int count_;

		void clone (const secure_buffer& t)
			{	verify (); 
				if (t.ptr_ == NULL) clear ();
				else { secure_buffer b (t); swap (b); } }
	public:
		secure_buffer () : ptr_ (NULL), count_ (0) { };
		secure_buffer (const secure_buffer& t) : ptr_ (NULL), count_ (t.count_)
			{	t.verify (); 
				if (count_ == 0) return;
				ptr_ = new T [count_]; 
				verify (); 
				::memcpy (ptr_, t.ptr_, size_in_bytes ()); }
		explicit secure_buffer (const unsigned int n) : ptr_ (NULL), count_ (n)
			{	if (count_ == 0) return;
				ptr_ = new T [count_]; 
				verify (); 
				::memset (ptr_, 0, size_in_bytes ()); }
		explicit secure_buffer (const T* const p, const unsigned int n) : ptr_ (NULL), count_ (n)
			{	if (count_ == 0) return;
				_ASSERT (! ::IsBadReadPtr (p, size_in_bytes ())); 
				ptr_ = new T [count_]; 
				verify (); 
				::memcpy (ptr_, p, size_in_bytes ()); }
		~secure_buffer () throw () { try { clear (); } catch (...) { } }
		secure_buffer& operator = (const secure_buffer& t) { if (&t != this) clone (t); return *this; }
		
		bool valid () const throw () { return ((count_ == 0) && (ptr_ == NULL)) || ((count_ != 0) && (! ::IsBadReadPtr (ptr_, size_in_bytes ()))); }
		bool valid () throw () { return ((count_ == 0) && (ptr_ == NULL)) || ((count_ != 0) && (! ::IsBadWritePtr (ptr_, size_in_bytes ()))); }
		void verify () const throw () { _ASSERT (valid ()); }
		void verify () throw () { _ASSERT (valid ()); } //lint !e1762
		unsigned int size_in_bytes () const throw () { _ASSERT ((count_ * sizeof (T)) < ULONG_MAX); return count_ * sizeof (T); }
			
		void swap (secure_buffer& t) throw ()
			{	verify (); t.verify (); 
				T* const p = ptr_; const unsigned int n = count_; 
				ptr_ = t.ptr_; count_ = t.count_; 
				t.ptr_ = p; t.count_ = n; }
		void clear ()
			{	verify (); 
				if (ptr_ != NULL) ::SecureZeroMemory (ptr_, size_in_bytes ());
				delete [] ptr_; 
				ptr_ = NULL; count_ = 0;  } 

		T* get () throw () { verify (); return ptr_; }
		const T* const get () const throw () { verify (); return ptr_; }
		unsigned int count () const throw () { return count_; }
		bool empty () const { return ptr_ == NULL; }
		bool operator ! () const throw () { return empty (); }
		void assign (const secure_buffer& t) { if (&t != this) clone (t); }
		void assign (const T* p, const unsigned int count) { secure_buffer b (p, count); swap (b); }
	};
	
}; //lint !e19
