#pragma once
#include "lip_except.h"
#include "lip_vest.h"
#include "lip_buffer.h"

namespace lip {
	
	class registry_key : boost::noncopyable
	{
	private:
		HKEY key_;
		bool nicked_;
		bool read_string (const std::wstring& value, std::wstring& reply, LONG& error) const;
		bool read_number (const std::wstring& value, unsigned int& reply, LONG& error) const;
	public:
		inline registry_key () throw () : key_ (NULL), nicked_ (false) { }
		explicit inline registry_key (const HKEY key, const bool nicked = true) throw () : key_ (key), nicked_ (nicked) { }
		explicit inline registry_key (const registry_key& parent, const std::wstring& name, const DWORD options = KEY_READ)
			: key_ (NULL), nicked_ (false) { init (parent, name, options); }
		~registry_key () throw () { try { clear (); } catch (...) { } key_ = NULL; }
		
		inline void release () throw () { key_ = NULL; }
		inline const HKEY data () const throw () { return key_; } //lint !e1763
		inline bool empty () const throw () { return (key_ == NULL); }

		void init (const registry_key& parent, const std::wstring& name, const DWORD options = KEY_READ)
			{	HKEY key = NULL;
				maybe_throw_error (::RegOpenKeyEx (parent.data (), name.c_str (), 0, options, &key));
				std::swap (key_, key); nicked_ = false; }
		void clear () {	if (key_ != NULL) if (! nicked_) maybe_throw_error (::RegCloseKey (key_)); key_ = NULL; }
		void remote_connect (const std::wstring& machine_name)
			{	_ASSERT (empty ()); _ASSERT (! machine_name.empty ());
				_ASSERT (machine_name [0] == L'\\'); _ASSERT (machine_name [1] == L'\\');
				HKEY key = NULL;
				maybe_throw_error (::RegConnectRegistry (machine_name.c_str (), HKEY_LOCAL_MACHINE, &key));
				std::swap (key_, key); nicked_ = false; }



		
		bool read_string (const std::wstring& value, std::wstring& reply) const
			{ LONG error; return read_string (value, reply, error); }
		std::wstring read_string (const std::wstring& value) const;
		HRESULT read_vest (const std::wstring& value, string_vest& vest) const;
		bool read_number (const std::wstring& value, unsigned int& reply) const
			{ LONG error; return read_number (value, reply, error); }
		unsigned int read_number (const std::wstring& value) const;
		bool enumerate (unsigned int& index, std::wstring& name) const;
	};
	
	
//	template <class T, bool integer_type> struct registry_readwrite {
//		static HRESULT read (const registry_key& key, const std::wstring& name, typename T& value);
//		static HRESULT write (const registry_key& key, const std::wstring& name, const typename T& value);
//	};
	
	template <class T, bool integer_type> struct registry_readwrite {
	
		static HRESULT read (const registry_key& key, const std::wstring& name, typename T& value)
			{	DWORD len = _MAX_PATH, type = REG_SZ;
				error = ::RegQueryValueEx (key_, value.c_str (), NULL, &type, NULL, &len);
				if (error != ERROR_SUCCESS) return MAKE_HRESULT (SEVERITY_ERROR, FACILITY_ITF, error);
				if (type != REG_SZ) return MAKE_HRESULT (SEVERITY_ERROR, FACILITY_ITF, ERROR_INVALID_DATA);
				temp_buffer <BYTE> buf (len / sizeof (wchar_t));
				error = ::RegQueryValueEx (key_, value.c_str (), NULL, NULL, buf.data (), &len);
				if (error != ERROR_SUCCESS) return MAKE_HRESULT (SEVERITY_ERROR, FACILITY_ITF, error);
				std::wostringstream s (std::wstring (buf.data ())); buf.clear ();
				s >> value; 
				return true; 
			}

		static HRESULT write (const registry_key& key, const std::wstring& name, const typename T& value)
			{	std::wistringstream s; s << value; DWORD len = (s.str ().length () + 1) * sizeof (wchar_t);
				error = ::RegSetValueEx (key, name.c_str (), NULL, REG_DWORD, reinterpret_cast <BYTE*> (s.str().c_str ()), len);
				if (error != ERROR_SUCCESS) return MAKE_HRESULT (SEVERITY_ERROR, FACILITY_ITF, error);
				return S_OK;
			}	//lint !e953
		
	};


	template <class T> struct registry_readwrite <T, true> {

		static HRESULT read (const registry_key& key, const std::wstring& name, typename T& value)
			{	DWORD n = 0; DWORD len = sizeof (DWORD), type = REG_DWORD;
				error = ::RegQueryValueEx (key, name.c_str (), NULL, &type, reinterpret_cast <BYTE*> (&n), &len);
				if (error != ERROR_SUCCESS) return MAKE_HRESULT (SEVERITY_ERROR, FACILITY_ITF, error);
				if (type != REG_DWORD) return MAKE_HRESULT (SEVERITY_ERROR, FACILITY_ITF, ERROR_INVALID_DATA);
				value = static_cast <typename T> (n); return S_OK;
			}	//lint !e953

		static HRESULT write (const registry_key& key, const std::wstring& name, const typename T& value)
				
			{	DWORD n = static_cast <DWORD> (value);
				error = ::RegSetValueEx (key, name.c_str (), NULL, REG_DWORD, reinterpret_cast <BYTE*> (&n), sizeof (DWORD));
				if (error != ERROR_SUCCESS) return MAKE_HRESULT (SEVERITY_ERROR, FACILITY_ITF, error);
				return S_OK;
			}	//lint !e953
	
	};
	
		
	template <class T> class registry_value : private boost::noncopyable
	{
	private:
		const registry_key& key_;
		const std::wstring name_;
		
	public:
		registry_value (const registry_key& key, const std::wstring& name) : key_ (key), name_ (name) 
			{ _ASSERT (! key_.empty ()); _ASSERT (! name_.empty ()); }
		HRESULT read (T& value) const
			{ return registry_readwrite <T, boost::traits::is_integral <T> >::read (key_, name_, value); }	
		HRESULT write (const T& value) const
			{ return registry_readwrite <T, boost::traits::is_integral <T> >::write (key_, name_, value); }
	};
	
	
}; //lint !e19
