#pragma once

#include "lip_except.h"


namespace lip {

	class service_manager : private boost::noncopyable

	{
	private:
		SC_HANDLE hSC_;
		SC_LOCK lock_;

	public:
		service_manager () : hSC_ (NULL), lock_ (NULL)
			{ hSC_ = ::OpenSCManager (NULL, NULL, SC_MANAGER_CONNECT); if (hSC_ == NULL) throw_last_error (); }
		~service_manager () throw () { try { clear (); } catch (...) { } hSC_ = NULL; }
		bool valid () const throw () { return hSC_ != NULL; }
		void clear () { if (valid ()) if (! ::CloseServiceHandle (hSC_)) throw_last_error (); hSC_ = NULL; }
		SC_HANDLE handle () const throw () { return hSC_; }
		bool locked () const throw () { return lock_ != NULL; }
	};


	class service_lock : private boost::noncopyable

	{
	private:
		const service_manager& manager_;
		const SC_LOCK lock_;

		service_lock ();
	public:
		explicit service_lock (const service_manager& manager)
				: manager_ (manager), lock_ (::LockServiceDatabase (manager_.handle ()))
			{	_ASSERT (manager_.valid ()); 
				if (lock_ == NULL) throw_last_error (); }
		~service_lock ()
			{	_ASSERT (lock_ != NULL); 
				_ASSERT (manager_.valid ()); 
				if (! ::UnlockServiceDatabase (manager_.handle ())) { /* do sod all */ };
			}
	};


	class service_control : private boost::noncopyable
	{
	public:
		typedef enum { indefinite, stopped, starting, stopping, running, restarting, pausing, paused, unknown } run_state;
		typedef enum { none, kernel, file, own, share, interact, illegal } service_type;
		typedef enum { still, boot, system, autostart, demand, disabled, never } service_start;
		typedef enum { broken, ignore, normal, severe, critical, explosive } error_control;
		
	private:
		SC_HANDLE hSvc_;
		SERVICE_STATUS_PROCESS status_;
		std::wstring name_;
		
		service_control ();
		void get_status ()
			{	_ASSERT (valid ());
				DWORD de;
				if (::QueryServiceStatusEx (hSvc_, 
											SC_STATUS_PROCESS_INFO,
											reinterpret_cast <LPBYTE> (&status_),
											sizeof (SERVICE_STATUS_PROCESS),
											&de) == FALSE)
					throw_last_error ();
			}
			
		static DWORD start_to_flags (const service_start start)
			{	_ASSERT ((start > still) && (start < never));
				return static_cast <DWORD> (start) - 1; 
			}
					
		static DWORD error_to_flags (const error_control err)
			{	_ASSERT ((err > broken) && (err < explosive));
				return static_cast <DWORD> (err) - 1; 
			}
					
		static DWORD type_to_flags (const service_type t)
			{	switch (t) {
					case kernel : return SERVICE_KERNEL_DRIVER;
					case file : return SERVICE_FILE_SYSTEM_DRIVER;
					case own : return SERVICE_WIN32_OWN_PROCESS;
					case share : return SERVICE_WIN32_SHARE_PROCESS;
					case interact : return SERVICE_WIN32_OWN_PROCESS | SERVICE_INTERACTIVE_PROCESS;
					default : break;
				}
				throw_error (ERROR_INVALID_PARAMETER);
			}
		
	public:
		service_control (const service_manager& manager, const std::wstring& name, const DWORD flags = SERVICE_ALL_ACCESS) 
				: hSvc_ (NULL), name_ (name)
			{	::memset (&status_, 0, sizeof (SERVICE_STATUS_PROCESS));
				_ASSERT (name_.length () > 0);
				_ASSERT (name_.length () <= 256);
				_ASSERT (name_.find (L'/') == std::wstring::npos);
				_ASSERT (name_.find (L'\\') == std::wstring::npos);
				_ASSERT (manager.valid ());
				hSvc_ = ::OpenService (manager.handle (), name_.c_str (), flags);
				if (hSvc_ == NULL) throw_last_error ();
				get_status ();
			}
		service_control (	const service_manager& manager, 
							const std::wstring& name, 
							const std::wstring& path,
							const std::wstring& description = std::wstring (),
							const service_type t = own,
							const service_start s = demand,
							const error_control e = normal)
				: hSvc_ (NULL), name_ (name)
			{	::memset (&status_, 0, sizeof (SERVICE_STATUS_PROCESS));
				_ASSERT (name_.length () > 0);
				_ASSERT (name_.length () <= 256);
				_ASSERT (name_.find (L'/') == std::wstring::npos);
				_ASSERT (name_.find (L'\\') == std::wstring::npos);
				_ASSERT (manager.valid ());
				service_lock lock (manager);
				hSvc_ = ::CreateService (	manager.handle (), 
											name_.c_str (), 
											description.c_str (), 
											SERVICE_ALL_ACCESS,
											type_to_flags (t),
											start_to_flags (s),
											error_to_flags (e),
											path.c_str (),
											NULL, NULL, NULL, NULL, NULL);
				if (hSvc_ != NULL) throw_last_error ();
				get_status ();
			}

		~service_control () throw () { try { clear (); } catch (...) { } hSvc_ = NULL; }
		bool valid () const throw () { return hSvc_ != NULL; }
		void clear () { if (valid ()) if (! ::CloseServiceHandle (hSvc_)) throw_last_error (); hSvc_ = NULL; }
		run_state state () const throw () { return static_cast <run_state> (status_.dwCurrentState); }
		void run ()
			{	_ASSERT (state () <= stopped);
				if (::StartService (hSvc_, 0, NULL) == FALSE) throw_last_error ();
				get_status();
			}
		void remove (const service_manager& manager)
			{	service_lock lock (manager);
				if (::DeleteService (hSvc_) == FALSE) throw_last_error ();
				get_status();
			}

	};
	
	
	class service_state
	{
	private:
		SERVICE_STATUS			status_;
		SERVICE_STATUS_HANDLE	h_;
	public:
		service_state () : h_ (NULL) { ::memset (&status_, 0, sizeof (SERVICE_STATUS)); }
/*		
		
    DWORD status; 
    DWORD specificError; 

SERVICE_STATUS          MyServiceStatus; 
SERVICE_STATUS_HANDLE   MyServiceStatusHandle; 
 
    MyServiceStatus.dwServiceType        = SERVICE_WIN32; 
    MyServiceStatus.dwCurrentState       = SERVICE_START_PENDING; 
    MyServiceStatus.dwControlsAccepted   = SERVICE_ACCEPT_STOP | 
        SERVICE_ACCEPT_PAUSE_CONTINUE; 
    MyServiceStatus.dwWin32ExitCode      = 0; 
    MyServiceStatus.dwServiceSpecificExitCode = 0; 
    MyServiceStatus.dwCheckPoint         = 0; 
    MyServiceStatus.dwWaitHint           = 0; 
 
    MyServiceStatusHandle = RegisterServiceCtrlHandler( 
        "MyService", 
        MyServiceCtrlHandler); 
 
    if (MyServiceStatusHandle == (SERVICE_STATUS_HANDLE)0) 
    { 
        SvcDebugOut(" [MY_SERVICE] RegisterServiceCtrlHandler 
            failed %d\n", GetLastError()); 
        return; 
    } 
 
    // Initialization code goes here. 
    status = MyServiceInitialization(argc,argv, &specificError); 
 
    // Handle error condition 
    if (status != NO_ERROR) 
    { 
        MyServiceStatus.dwCurrentState       = SERVICE_STOPPED; 
        MyServiceStatus.dwCheckPoint         = 0; 
        MyServiceStatus.dwWaitHint           = 0; 
        MyServiceStatus.dwWin32ExitCode      = status; 
        MyServiceStatus.dwServiceSpecificExitCode = specificError; 
 
        SetServiceStatus (MyServiceStatusHandle, &MyServiceStatus); 
        return; 
    } 
 
    // Initialization complete - report running status. 
    MyServiceStatus.dwCurrentState       = SERVICE_RUNNING; 
    MyServiceStatus.dwCheckPoint         = 0; 
    MyServiceStatus.dwWaitHint           = 0; 
 
    if (!SetServiceStatus (MyServiceStatusHandle, &MyServiceStatus)) 
    { 
        status = GetLastError(); 
        SvcDebugOut(" [MY_SERVICE] SetServiceStatus error
            %ld\n",status); 
    } 
 
    // This is where the service does its work. 
    SvcDebugOut(" [MY_SERVICE] Returning the Main Thread \n",0); 
*/ 
	
	};
	
}; //lint !e19
