/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence,  or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#pragma once

#include <iostream>
#include <string>
#include <vector>
#if defined (FUDDYDUDDY)
// at least macos 10.13
#include <set>
#include <map>
#define ssc_set ::std::set
#define ssc_map ::std::map
#else // FUDDYDUDDY
#include <unordered_set>
#include <unordered_map>
#define ssc_set ::std::unordered_set
#define ssc_map ::std::unordered_map
#endif // FUDDYDUDDY
#include <boost/filesystem.hpp>
#include <boost/property_tree/ptree.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/lexical_cast.hpp>

#define WEBMENTION "webmention"
#define SEP "."

#define HTTP_PROTOCOL "http"
#define HTTPS_PROTOCOL "https"
#define CSS "://"
#define HTTP HTTP_PROTOCOL CSS
#define HTTPS HTTPS_PROTOCOL CSS
#define SPACE ' '
#define COLON ':'
#define DOT '.'
#define SLASH '/'
#define HASH '#'
#define QUESTION '?'
#define AT '@'
#define EQUAL '='
#define PERCENT '%'
#define BRCLOSE ')'
#define BROPEN '('
#define SQCLOSE ']'
#define SQOPEN '['
#define CUCLOSE '}'
#define CUOPEN '{'
#define DENERY "0123456789"
#define DDD DENERY "-."
#define DECIMAL DDD "+"
#define OCTAL "01234567"
#define HEX "0123456789abcdefABCDEF"
#define ALPHABET "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
#define SPACESTEP "  "
#define FYI SPACESTEP SPACESTEP "> "

#define DEFAULT_DOMAIN "example.org"

const int STEPSPACES = 2;

template < typename TYPE > struct lexical
{   static bool test (const ::std::string& s)
    {   if (s.empty ()) return true;
        try
        {   ::boost::lexical_cast < TYPE > (s); return true; }
        catch (...)
        {   return false; } }
    static TYPE cast (const ::std::string& s)
    {   try
        {   return ::boost::lexical_cast < TYPE > (s); }
        catch (...)
        {   return TYPE (); } }
    static TYPE cast (const ::std::string& s, const TYPE def)
    {   try
        {   return ::boost::lexical_cast < TYPE > (s); }
        catch (...)
        {   return def; } }
    static TYPE cast2 (const ::std::string& s, bool& b)
    try
    {   b = true;
        return ::boost::lexical_cast < TYPE > (s); }
    catch (...)
    {   b = false;
        return TYPE (); } };



// template paramas: new, changed, deleted, unchanged, unknown
typedef enum { act_insert, act_update, act_delete, act_static, act_unknown } e_activity;
typedef ::std::vector < ::std::string > vstr_t;
typedef ssc_set < ::std::string > sstr_t;
typedef ssc_map < ::std::string, ::std::string > ustr_t;
typedef ustr_t::value_type ustrv_t;

::std::string trim_the_lot_off (const ::std::string& s);
bool remove_tail (::std::string& s, ::std::string& tail, const char ch);
bool remove_head (::std::string& s, ::std::string& head, const char ch);
::std::string remove_tail (::std::string& s, const char ch);
::std::string remove_head (::std::string& s, const char ch);
bool separate (const ::std::string& s, ::std::string& head, ::std::string& tail, const char ch);

vstr_t split_by_charset (const ::std::string& s, const char* charset);
::std::string read_text_file (const ::std::string& name);
bool write_text_file (const ::std::string& name, const ::std::string& content);
::boost::filesystem::path get_tmp_filename ();

inline bool cnc_test (unsigned char a, unsigned char b)
{ return ::std::tolower (a) == ::std::tolower (b); }

bool compare_no_case (const std::string& a, const std::string& b);
bool is_one_of (const ::std::string& s, const vstr_t& v);

inline vstr_t split_by_space (const ::std::string& s)
{   return split_by_charset (s, " "); }

inline vstr_t split_by_newline (const ::std::string& s)
{   return split_by_charset (s, "\n"); }

template < class T > T read_field (::boost::property_tree::ptree& tree, const char* field)
{   return tree.get (field, T ()); }

template < class T > T read_field (::boost::property_tree::ptree& tree, const ::std::string& container, const char* field)
{   ::std::string name (container);
    name += SEP;
    name += field;
    return tree.get (name, T ()); }

template < class T > void write_field (::boost::property_tree::ptree& tree, const char* field, const T& value)
{   tree.put (field, value); }

template < class T > void write_field (::boost::property_tree::ptree& tree, const ::std::string& container, const char* field, const T& value)
{   ::std::string name (container);
    name += SEP;
    name += field;
    tree.put (name, value); }

bool read_header (::boost::property_tree::ptree& json, const ::std::string& expected, ::std::string& version, const ::std::string& filename);
void write_header (::boost::property_tree::ptree& json, const char* context);
bool replace_file (::boost::property_tree::ptree& json, ::boost::filesystem::path& filename);

inline ::std::string slash_dot (const ::std::string& slash)
{   ::std::string dot (slash);
    for (auto i = dot.begin (); i != dot.end (); ++i)
        if (*i == '/') *i = '.';
    return dot; }

inline ::std::string fyi (const int depth)
{   ::std::string res (STEPSPACES*depth, ' ');
    res += FYI;
    return res; }

inline void append (::std::string& base, const ::std::string& sep, const ::std::string& xtr)
{   if (! xtr.empty ())
    {   if (! base.empty ()) base += sep;
        base += xtr; }}

inline void prepend (::std::string& base, const ::std::string& sep, const ::std::string& xtr)
{   if (! xtr.empty ())
        if (base.empty ()) base = xtr;
        else base = xtr + sep + base; }
